package com.atlassian.distribution.scriptwriter;


import org.apache.commons.io.FileUtils;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

public abstract class AbstractScriptWriter {
    protected String filename;
    protected File directory;
    protected FileWriter fileWriter;

    protected List<String> preliminaryLines = new ArrayList<String>();
    protected List<String> commands = new ArrayList<String>();
    protected String betweenEachCommand = "";
    protected String newLine = "";
    protected String appendToMavenCommands = "";
    protected String extraCli = "";

    public AbstractScriptWriter(File directory, String filename, boolean withTests) {
        this.filename = filename;
        this.directory = directory;
        if (!withTests) {
            this.appendToMavenCommands = " -Dmaven.test.skip";
        }
    }

    /**
     * Writes the stored script info out to file
     *
     * @throws java.io.IOException
     */
    public void writeScript() throws IOException {
        this.useErrorMode();
        this.addCurrentDirToPath();
        File scriptFile = new File(directory, filename);
        this.fileWriter = new FileWriter(scriptFile);
        for (String preliminary : preliminaryLines) {
            fileWriter.write(preliminary + newLine);
        }

        for (String command : commands) {
            fileWriter.write(command + newLine);
            if (isNotBlank(betweenEachCommand)) {
                fileWriter.write(betweenEachCommand + newLine);
            }
        }

        scriptFile.setExecutable(true, false);
        fileWriter.flush();
        fileWriter.close();
    }

    protected void addToMavenCommands(String extraArgs) {
        this.appendToMavenCommands = this.appendToMavenCommands + " " + extraArgs;
    }

    /**
     * Copies a maven settings file for inclusion in the source distribution.
     *
     * @param sourceFilename
     * @param destFilename
     * @throws java.io.IOException
     */
    public void includeSettingsFile(String sourceFilename, String destFilename) throws IOException {
        FileUtils.copyFile(new File(sourceFilename), new File(this.directory, destFilename));
    }


    /**
     * Adds a "-s settingsFilename" to all maven commands
     *
     * @param settingsFilename
     */
    abstract public void addSettingsLine(String settingsFilename);

    /**
     * Adds "-Dmaven.repo.local=localRepoFilename" to all maven commands
     */
    abstract public void addLocalRepoLine(String localRepoFilename);

    /**
     * Adds a command to put the current directory in the system path
     */
    abstract public void addCurrentDirToPath();

    /**
     * Makes the script halt on the first error rather than plowing on
     */
    abstract public void useErrorMode();

    /**
     * Allows adding of extra commands before the main project commands
     *
     * @param line
     */
    abstract public void addOtherPreliminaryLine(String line);

    /**
     * Adds a command without any arguments
     *
     * @param command
     * @param isMaven
     * @param takesExtraCli
     */
    abstract public void addCommand(String command, boolean isMaven, boolean takesExtraCli);

    /**
     * Adds a command with arguments
     *
     * @param command
     * @param arguments
     * @param isMaven
     * @param takesExtraCli
     */
    abstract public void addCommand(String command, String arguments, boolean isMaven, boolean takesExtraCli);
}
