package com.atlassian.maven.plugins.pdk;

import java.io.BufferedReader;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;

/**
 * @goal uninstall
 * @description Uninstall a plugin from confluence
 */
public class UninstallPluginMojo extends BasePluginServerMojo {
    private static final String UPLOAD = "upload";

    private static final String COPY = "copy";

    /**
     * @parameter expression="${atlassian.pdk.install.method}"
     *            default-value="upload"
     * @required
     */
    protected String installMethod = UPLOAD;

    /**
     * @parameter expression="${atlassian.pdk.server.home}/plugins"
     */
    protected File pluginsHome;

    /**
     * @parameter expression="${project.artifactId}"
     */
    protected String artifactId;

    protected String getTitle() {
        return "Uninstall Plugin";
    }
    
    protected void init( String user, String pass, String confUrl, String pluginKey, String installMethod, File pluginsHome ) {

        this.username = user;
        this.password = pass;
        this.serverUrl = confUrl;
        this.pluginKey = pluginKey;
        this.installMethod = installMethod;
        this.pluginsHome = pluginsHome;
    }

    protected boolean checkProperties() {

        if ( installMethod == null
                || ( !UPLOAD.equalsIgnoreCase( installMethod ) && !COPY.equalsIgnoreCase( installMethod ) ) ) {
            getLog().error(
                    "Please specify either '" + UPLOAD + "' or '" + COPY
                            + "' for the 'atlassian.pdk.install.method' property." );
            return true;
        }

        if ( COPY.equals( installMethod ) && pluginsHome == null ) {
            getLog().error( "Please specify the 'atlassian.pdk.server.home' property." );
            return true;
        }

        return super.checkProperties();
    }

    public void execute() throws MojoFailureException, MojoExecutionException {

        if ( checkProperties() )
            return;

        if ( COPY.equals( installMethod ) ) {
            // Copy Plugin to plugin Home Folder
            deleteFiles( artifactId, pluginsHome );
        } else if ( UPLOAD.equals( installMethod ) ) {
            super.execute();
        }
    }

    private void deleteFiles( final String artifactId, File pluginsHome ) {
        File[] files = pluginsHome.listFiles( new FilenameFilter() {
            public boolean accept( File dir, String name ) {
                return name.startsWith( artifactId + "-" ) && name.endsWith( ".jar" );
            }
        } );

        if ( files != null && files.length > 0 ) {
            for ( int i = 0; i < files.length; i++ ) {
                if ( files[i].exists() && checkPluginKey( files[i] ) && files[i].delete() ) {
                    getLog().info( getTitle() + ": Deleted previous version of plugin: " + files[i].getName() );
                } else {
                    getLog().info( getTitle() + ": Did not delete existing jar file: " + files[i].getName() );
                }
            }
        } else {
            getLog().info( getTitle() + ": No previous versions of plugin were found." );
        }
    }

    private boolean checkPluginKey( File file ) {
        BufferedReader in = null;
        try {
            JarFile jar = new JarFile( file );
            JarEntry apx = jar.getJarEntry( "atlassian-plugin.xml" );
            if ( apx != null ) {
                in = new BufferedReader( new InputStreamReader( jar.getInputStream( apx ) ) );
                String line = null;
                while ( (line = in.readLine()) != null && line.indexOf( pluginKey ) < 0 );
                
                // Return true if the current line is not null - i.e. it contains the plugin key. 
                boolean matches = line != null;
                if ( !matches )
                    getLog().info( getTitle() + ": The plugin specified does not match the required plugin key: " + pluginKey );
                return matches;
            }
        } catch ( IOException e ) {
            getLog().warn( getTitle() + ": Unable to read jar file: " + file.getName(), e );
        } finally {
            if ( in != null ) {
                try {
                    in.close();
                } catch ( IOException e ) {
                    getLog().warn( getTitle() + ": Error while closing the atlassian-plugin.xml stream", e );
                }
            }
        }
        
        return false;
    }

    protected String getMode() {
        return "uninstall";
    }
}