package com.atlassian.maven.plugins.pdk;

import com.google.common.collect.ImmutableMap;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientHandlerException;
import com.sun.jersey.api.client.ClientRequest;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import com.sun.jersey.api.client.filter.ClientFilter;
import com.sun.jersey.api.client.filter.LoggingFilter;
import com.sun.jersey.client.urlconnection.HTTPSProperties;
import com.sun.jersey.multipart.FormDataMultiPart;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.ConnectException;
import java.net.URI;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.MultipartPostMethod;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.json.JSONException;
import org.json.JSONObject;

import static java.nio.file.Files.readAllBytes;
import static javax.ws.rs.core.HttpHeaders.AUTHORIZATION;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON_TYPE;
import static javax.ws.rs.core.MediaType.APPLICATION_OCTET_STREAM_TYPE;
import static javax.ws.rs.core.MediaType.MULTIPART_FORM_DATA_TYPE;
import static javax.ws.rs.core.Response.Status.ACCEPTED;
import static javax.ws.rs.core.Response.Status.OK;

/**
 * @goal install
 * @execute phase="package"
 * @description Install a plugin into Confluence
 */
public class InstallPluginMojo extends BasePluginServerMojo {

    private static final String UPLOAD = "upload";

    private static final String COPY = "copy";

    public static final String PENDING_TASK_JSON = "application/vnd.atl.plugins.pending-task+json";
    public static final String UPM_ROOT_RESOURCE = "/rest/plugins/1.0/";

    /**
     * @parameter expression="${atlassian.pdk.install.method}"
     *            default-value="upload"
     * @required
     */
    protected String installMethod = UPLOAD;

    /**
     * @parameter expression="${atlassian.pdk.server.home}/plugins"
     */
    protected File pluginsHome;

    /**
     * @parameter expression="${project.build.directory}/${project.build.finalName}.jar"
     * @required
     */
    protected File pluginFile;

    /**
     * @parameter expression="${project.build.directory}/${project.build.finalName}-signature.txt"
     */
    protected File pluginSignatureFile;

    protected String getTitle() {

        return "Install Plugin";
    }

    protected boolean checkProperties() {

        if ( installMethod == null
             || ( !UPLOAD.equalsIgnoreCase( installMethod ) && !COPY.equalsIgnoreCase( installMethod ) ) ) {
            getLog().error(
                "Please specify either '" + UPLOAD + "' or '" + COPY
                + "' for the 'atlassian.pdk.install.method' property." );
            return true;
        }

        if ( COPY.equals( installMethod ) && pluginsHome == null ) {
            getLog().error( "Please specify the 'atlassian.pdk.server.home' property." );
            return true;
        }

        if ( pluginFile == null ) {
            getLog().error( "The plugin file could not be found." );
            return true;
        } else if ( !pluginFile.exists() || !pluginFile.isFile() ) {
            getLog().error( "The required plugin file does not exist: " + pluginFile.getAbsolutePath() );
            return true;
        }

        return super.checkProperties();
    }

    public void execute() throws MojoFailureException, MojoExecutionException {

        if ( checkProperties() )
            return;

//        // Uninstall Plugin
//        UninstallPluginMojo up = new UninstallPluginMojo();
//        up.init( username, password, serverUrl, pluginKey, installMethod, pluginsHome );
//        up.execute();

        if ( COPY.equals( installMethod ) ) {
            // Copy Plugin to plugin Home Folder
            copyFiles( pluginFile, pluginsHome );
            getLog().info( getTitle() + ": Copied the plugin to '" + pluginsHome + "'" );

            // Rescan for Plugins
            RescanPluginsMojo rp = new RescanPluginsMojo();
            rp.init( username, password, serverUrl, pluginKey );
            rp.execute();
        } else if ( UPLOAD.equals( installMethod ) ) {
            uploadFile();
        }
    }

    private void uploadFile() throws MojoExecutionException {
        ClientConfig config = new DefaultClientConfig();
        configureSSL(serverUrl, config);
        Client client = Client.create(config);
        if (getLog().isDebugEnabled())
        {
            client.addFilter(new LoggingFilter());
        }
        client.addFilter(new BasicAuthFilter(username, password));
        client.setFollowRedirects(false);

        URI uri = URI.create(serverUrl + UPM_ROOT_RESOURCE);
        final FormDataMultiPart pluginPart = new FormDataMultiPart();
        pluginPart.field("plugin", pluginFile, APPLICATION_OCTET_STREAM_TYPE);

        // Signature file is optional. Note, AMPS sets a default signature file value; The default value might point to a non-existing file.
        if (pluginSignatureFile != null && pluginSignatureFile.exists()) {
            updateSignatureField(pluginPart);
        } else {
            getLog().info("The plugin signature file was not found. Signature check must be disabled on the server.");
        }

        final MediaType multipartFormDataWithBoundary =
            new MediaType(MULTIPART_FORM_DATA_TYPE.getType(),
                          MULTIPART_FORM_DATA_TYPE.getSubtype(),
                          ImmutableMap.<String, String>builder().put("boundary", "some simple boundary").build());

        WebResource resource = client.resource(uri.toASCIIString());

        String token = "badtoken";
        ClientResponse tokenResponse = resource.head();
        token = tokenResponse.getMetadata().getFirst("upm-token");

        if (token == null) {
            getLog().info( getTitle() + ": Couldn't get the token from upm. Headers. Falling back to legacy upload.");
            legacyUploadFile();
            return;
        }

        getLog().info( getTitle() + ": Uploading '" + pluginFile.getName() + "' to server via UPM: " + serverUrl );
        ClientResponse acceptResponse = resource
            .queryParam("token", token)
            .header("X-Atlassian-Token", "no-check")
            .type(multipartFormDataWithBoundary)
            .post(ClientResponse.class, pluginPart);

        try
        {
            waitForCompletion(client, acceptResponse);
        }
        catch (AsynchronousTaskException ate)
        {
            throw new MojoExecutionException(getTitle() + " : Upload failed --- " + ate.getMessage());
        }
    }

    private void updateSignatureField(FormDataMultiPart formDataMultiPart) throws MojoExecutionException {
        try {
            String signatureStr = new String(readAllBytes(pluginSignatureFile.toPath())).replace("\n", "");
            java.util.Base64.getDecoder().decode(signatureStr); // test if it's a valid base64 string
            formDataMultiPart.field("signature", String.format("{\"signature\":\"%s\"}", signatureStr), APPLICATION_JSON_TYPE);
        } catch (IOException e) {
            throw new MojoExecutionException("Failed to read signature file: " + pluginSignatureFile, e);
        } catch (IllegalArgumentException e) {
            throw new MojoExecutionException("Invalid signature. It must be base64 encoded. File: " + pluginSignatureFile, e);
        }
    }

    /**
     * This is necessary for applications that don't embed UPM (i.e., Crowd).
     *
     * @throws MojoExecutionException
     */
    private void legacyUploadFile() throws MojoExecutionException {

        // Create an instance of HttpClient.
        HttpClient client = new HttpClient();

        // Create a method instance.
        String url = serverUrl + "/admin/uploadplugin.action" +"?os_username=" + urlEncode(username) + "&os_password=" + urlEncode(password);

        MultipartPostMethod filePost = new MultipartPostMethod( url );
        filePost.setDoAuthentication(true);
        filePost.setFollowRedirects( true );

        // bypass anti xsrf protection
        filePost.setRequestHeader("X-Atlassian-Token", "no-check");

        try {
            filePost.addParameter( "file_0", pluginFile.getName(), pluginFile );
            client.setConnectionTimeout( 5000 );
            // Execute the method.

            getLog().info( getTitle() + ": Uploading '" + pluginFile.getName() + "' to server: " + serverUrl );
            int status = client.executeMethod(filePost.getHostConfiguration(), filePost , getHttpState());

            if ( status == HttpStatus.SC_MOVED_TEMPORARILY ) {
                Header location = filePost.getResponseHeader( "Location" );
                if ( location == null ) {
                    throw new MojoExecutionException( getTitle() + ": Upload failed[" + status
                            + "]: Redirecing to an unknown location" );
                } else if ( location.getValue().indexOf( "/login.action" ) >= 0 )
                    throw new MojoExecutionException( getTitle() + ": Upload failed[" + status
                            + "]: This is likely due to a non-existent or underpriviledged user: " + username );
            } else if ( status != HttpStatus.SC_OK ) {

                throw new MojoExecutionException( getTitle() + ": Upload failed[" + status + "]: "
                        + HttpStatus.getStatusText( status ) );
            }
            getLog().info( getTitle() + ": Completed successfully[" + status + "]." );
        } catch ( ConnectException e ) {
            getLog().error( getTitle() + ": Unable to connect to '" + serverUrl + "': " + e.getMessage() );
            // getLog().debug(e);
        } catch ( FileNotFoundException e ) {
            getLog().error( getTitle() + ": Unable to find file to upload: " + e.getMessage() );
        } catch ( HttpException e ) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch ( IOException e ) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } finally {
            filePost.releaseConnection();
        }
    }

    private void configureSSL(String url, ClientConfig config) throws MojoExecutionException
    {
        if (url.startsWith("https:"))
        {
            return;
        }

        SSLContext ctx;
        try
        {
            ctx = SSLContext.getInstance("SSL");
        }
        catch (NoSuchAlgorithmException e)
        {
            throw new MojoExecutionException("Can't initialise the SSL context", e);
        }
        // Trust anything (insecure - the risk is for the plugin.jar to be intercepted by a MITM)
        TrustManager[] myTrustManager = { new X509TrustManager()
        {
            @Override
            public void checkClientTrusted(X509Certificate[] arg0, String arg1) throws CertificateException
            {
                // Yeah, you're good mate!
            }

            @Override
            public void checkServerTrusted(X509Certificate[] arg0, String arg1) throws CertificateException
            {
                // No worries. You're just uploading a plugin ey?
            }

            @Override
            public X509Certificate[] getAcceptedIssuers()
            {
                // Anyone is accepted
                return null;
            }
        } };
        try
        {
            ctx.init(null, myTrustManager, null);
        }
        catch (KeyManagementException e)
        {
            throw new MojoExecutionException("Can't initialise the SSL context", e);
        }

        HostnameVerifier hostnameVerifier = new HostnameVerifier()
        {
            @Override
            public boolean verify(String hostname, SSLSession session)
            {
                return true;
            }
        };
        config.getProperties().put(HTTPSProperties.PROPERTY_HTTPS_PROPERTIES, new HTTPSProperties(hostnameVerifier, ctx));
    }

    private static class AsynchronousTaskException extends RuntimeException
    {

        public AsynchronousTaskException(String s, Throwable cause)
        {
            super(s, cause);
        }
        public AsynchronousTaskException(String s)
        {
            super(s);
        }
    }

    public void waitForCompletion(Client client, ClientResponse acceptResponse)
    {
        URI taskLocation = acceptResponse.getLocation();

        WebResource taskResource = client.resource(taskLocation);

        ClientResponse clientResponse = null;
        // set our limit to only 30 iterations before going out of the loop to prevent a possible infinite loop.
        for (int i = 0; i < 30; i++)
        {
            clientResponse = taskResource.accept(PENDING_TASK_JSON).get(ClientResponse.class);

            if (isAdditionalTaskRequested(clientResponse))
            {
                String entity = clientResponse.getEntity(String.class);

                try
                {
                    executeNextTask(client, new JSONObject(entity));
                    break;
                }
                catch (JSONException e)
                {
                    throw new AsynchronousTaskException("Failed to parse JSON : " + entity, e);
                }
            }
            if (clientResponse.getStatus() != OK.getStatusCode())
            {
                break;
            }
            if (isTaskErrorRepresentation(clientResponse))
            {
                throw new AsynchronousTaskException(clientResponse.getEntity(String.class));
            }
            if (isTaskDone(clientResponse))
            {
                break;
            }
            try // sleep for a second, and retry to see if the task has completed
            {
                Thread.sleep(1000);
            }
            catch (InterruptedException ex)
            {
            }
        }
    }

    private void executeNextTask(Client client, JSONObject object) throws JSONException
    {
        JSONObject status = object.getJSONObject("status");

        if (status == null)
        {
            throw new AsynchronousTaskException("Cannot parse status from : " + object );
        }

        if (status.has("nextTaskPostUri"))
        {
            getLog().info( getTitle() + ": Triggering UPM self-update");
            WebResource resource = client.resource(status.getString("nextTaskPostUri"));
            ClientResponse response = resource.post(ClientResponse.class);
            waitForCompletion(client, response);
        }
        else if (status.has("cleanupDeleteUri"))
        {
            getLog().info( getTitle() + ": Uninstalling UPM self-update stub plugin");
            WebResource resource = client.resource(status.getString("cleanupDeleteUri"));
            resource.delete(ClientResponse.class);
        }
        else
        {
            throw new AsynchronousTaskException("status should contain nextTaskPostUri or nextTaskDeleteUri, but was: " + status );
        }
    }

    private boolean isAdditionalTaskRequested(ClientResponse response)
    {
        return response.getStatus() == ACCEPTED.getStatusCode() && response.getType().getSubtype().endsWith("next-task+json");
    }

    private boolean isTaskErrorRepresentation(ClientResponse response)
    {
        return response.getType().getSubtype().endsWith("err+json");
    }

    private boolean isTaskDone(ClientResponse response)
    {
        return response.getType().getSubtype().endsWith("complete+json");
    }

    final class BasicAuthFilter extends ClientFilter
    {
        private final String auth;

        BasicAuthFilter(String username, String password)
        {
            try
            {
                auth = "Basic " + new String(Base64.encodeBase64((username + ":" + password).getBytes("ASCII")));
            }
            catch (UnsupportedEncodingException e)
            {
                throw new RuntimeException("That's some funky JVM you've got there", e);
            }
        }

        public ClientResponse handle(ClientRequest cr) throws ClientHandlerException
        {
            MultivaluedMap<String, Object> headers = cr.getMetadata();
            if (!headers.containsKey(AUTHORIZATION))
            {
                headers.add(AUTHORIZATION, auth);
            }
            return getNext().handle(cr);
        }
    }

    protected String getMode() {
        return null;
    }
}
