package com.atlassian.maven.plugins.pdk;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.resolver.filter.ScopeArtifactFilter;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.archiver.ArchiverException;
import org.codehaus.plexus.archiver.UnArchiver;
import org.codehaus.plexus.archiver.manager.ArchiverManager;
import org.codehaus.plexus.archiver.manager.NoSuchArchiverException;
import org.codehaus.plexus.util.DirectoryScanner;
import org.codehaus.plexus.util.FileUtils;
import org.codehaus.plexus.util.StringUtils;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * <i>Code borrowed liberally from the standard Maven 2 'war' plugin.</i>
 *
 * @goal bundle-jars
 * @phase process-classes
 * @requiresDependencyResolution runtime
 */
public class BundleJarsMojo extends BasePdkMojo
{
    private static final String META_INF = "META-INF";
    private static final String DEFAULT_EXCLUDES =
        "META-INF/*.SF,META-INF/*.sf,META-INF/*.DSA,META-INF/*.dsa,META-INF/*.RSA,META-INF/*.rsa";


    /**
     * The maven project.
     *
     * @parameter expression="${project}"
     * @required
     * @readonly
     */
    private MavenProject project;

    /**
     * @parameter expression="${extractDependencies}" default-value="false"
     */
    private boolean extractDependencies;

    /**
     * @parameter expression="${settings.localRepository}"
     */
    private String localRepository;

    /**
     * @parameter expression="${project.build.directory}/classes"
     */
    private String target;

    /**
     * Directory to unpack dependencies into if needed
     *
     * @parameter expression="${project.build.directory}/bundle-jars/work"
     * @required
     */
    private File workDirectory;

    /**
     * The comma separated list of tokens to include when doing an overlay.
     * Default is '**'.
     *
     * @parameter
     */
    private String dependencyIncludes = "**";

    /**
     * The comma separated list of tokens to exclude when doing an overlay.
     * Default is 'META-INF/**'.
     *
     * @parameter
     */
    private String dependencyExcludes = DEFAULT_EXCLUDES;

    /**
     * To look up Archiver/UnArchiver implementations
     *
     * @parameter expression="${component.org.codehaus.plexus.archiver.manager.ArchiverManager}"
     * @required
     */
    protected ArchiverManager archiverManager;

    private static final String[] EMPTY_STRING_ARRAY = {};

    public void execute() throws MojoExecutionException, MojoFailureException
    {
        File destFile = new File(target);
        try
        {
            bundleDependencies(project, destFile);
        }
        catch (IOException e)
        {
            throw new MojoExecutionException("Failure while building project", e);
        }
    }

    /**
     * Bundles the required jars the specified project.
     * <p/>
     * Classes, libraries and tld files are copied to
     * the <tt>buildDir</tt> during this phase.
     *
     * @param project  the maven project
     * @param buildDir The temporary atlassian directory for building the plugin.
     * @throws java.io.IOException if an error occured while building the plugin
     * @throws org.apache.maven.plugin.MojoFailureException
     *                             if there is a failure.
     * @throws org.apache.maven.plugin.MojoExecutionException
     *                             if there is a problem while executing.
     */
    public void bundleDependencies(MavenProject project, File buildDir)
        throws MojoExecutionException, IOException, MojoFailureException
    {
        getLog().info("Bundling dependencies for '" + project.getArtifactId() + "' in " + buildDir);

        File metainfDir = new File(buildDir, META_INF);
        metainfDir.mkdirs();

        File libDirectory = new File(metainfDir, "lib");

        Set artifacts = project.getArtifacts();

        List duplicates = findDuplicates(artifacts);

        List dependencyDirectories = new ArrayList();

        for (Iterator iter = artifacts.iterator(); iter.hasNext();)
        {
            Artifact artifact = (Artifact) iter.next();
            String targetFileName = getDefaultFinalName(artifact);

            getLog().debug("Processing: " + targetFileName);

            if (duplicates.contains(targetFileName))
            {
                getLog().debug("Duplicate found: " + targetFileName);
                targetFileName = artifact.getGroupId() + "-" + targetFileName;
                getLog().debug("Renamed to: " + targetFileName);
            }

            // TODO: utilise appropriate methods from project builder
            ScopeArtifactFilter runtimeFilter = new ScopeArtifactFilter(Artifact.SCOPE_RUNTIME);
            ScopeArtifactFilter compileFilter = new ScopeArtifactFilter(Artifact.SCOPE_COMPILE);
            if (!artifact.isOptional() && (runtimeFilter.include(artifact) || compileFilter.include(artifact)))
            {
                String type = artifact.getType();
                if ("jar".equals(type) || "ejb".equals(type) || "ejb-client".equals(type))
                {
                    getLog().info("Bundling '" + artifact.getArtifactId() + "' into plugin.");
                    if (extractDependencies)
                        dependencyDirectories.add(unpackToTempDirectory(artifact));
                    else
                        copyFileIfModified(artifact.getFile(), new File(libDirectory, targetFileName));
                }
                else
                {
                    if ("par".equals(type))
                    {
                        targetFileName = targetFileName.substring(0, targetFileName.lastIndexOf('.')) + ".jar";

                        getLog().debug(
                            "Copying " + artifact.getFile() + " to " + new File(libDirectory, targetFileName));

                        getLog().info("Bundling '" + artifact.getArtifactId() + "' into plugin.");
                        if (extractDependencies)
                            dependencyDirectories.add(unpackToTempDirectory(artifact));
                        else
                            copyFileIfModified(artifact.getFile(), new File(libDirectory, targetFileName));
                    }
                    else
                    {
                        if ("war".equals(type))
                        {
                            dependencyDirectories.add(unpackToTempDirectory(artifact));
                        }
                        else
                        {
                            getLog().info("Not bundling '" + artifact.getArtifactId() + "' - unsupported type: " + type);
                        }
                    }
                }
            }
        }

        if (dependencyDirectories.size() > 0)
        {
            getLog().info("Overlaying " + dependencyDirectories.size() + " dependencies.");

            // overlay dependencies
            for (Iterator iter = dependencyDirectories.iterator(); iter.hasNext();)
            {
                copyDependencyContents((File) iter.next(), buildDir);
            }
        }
    }

    /**
     * Searches a set of artifacts for duplicate filenames and returns a list of duplicates.
     *
     * @param artifacts set of artifacts
     * @return List of duplicated artifacts
     */
    private List findDuplicates(Set artifacts)
    {
        List duplicates = new ArrayList();
        List identifiers = new ArrayList();
        for (Iterator iter = artifacts.iterator(); iter.hasNext();)
        {
            Artifact artifact = (Artifact) iter.next();
            String candidate = getDefaultFinalName(artifact);
            if (identifiers.contains(candidate))
            {
                duplicates.add(candidate);
            }
            else
            {
                identifiers.add(candidate);
            }
        }
        return duplicates;
    }

    /**
     * Converts the filename of an artifact to artifactId-version.type format.
     *
     * @param artifact The artifact
     * @return converted filename of the artifact
     */
    private String getDefaultFinalName(Artifact artifact)
    {
        return artifact.getArtifactId() + "-" + artifact.getVersion() + "." +
            artifact.getArtifactHandler().getExtension();
    }

    /**
     * Copy file from source to destination only if source timestamp is later than the destination timestamp.
     * The directories up to <code>destination</code> will be created if they don't already exist.
     * <code>destination</code> will be overwritten if it already exists.
     *
     * @param source      An existing non-directory <code>File</code> to copy bytes from.
     * @param destination A non-directory <code>File</code> to write bytes to (possibly
     *                    overwriting).
     * @throws IOException                   if <code>source</code> does not exist, <code>destination</code> cannot be
     *                                       written to, or an IO error occurs during copying.
     * @throws java.io.FileNotFoundException if <code>destination</code> is a directory
     *                                       <p/>
     *                                       TO DO: Remove this method when Maven moves to plexus-utils version 1.4
     */
    private static void copyFileIfModified(File source, File destination)
        throws IOException
    {
        // TO DO: Remove this method and use the method in WarFileUtils when Maven 2 changes
        // to plexus-utils 1.2.
        if (destination.lastModified() < source.lastModified())
        {
            FileUtils.copyFile(source.getCanonicalFile(), destination);
            // preserve timestamp
            destination.setLastModified(source.lastModified());
        }
    }

    /**
     * Unpacks dependency artifacts into a temporary directory inside <tt>workDirectory</tt>
     * named with the name of the dependency file.
     *
     * @param artifact artifact to unpack.
     * @return Directory containing the unpacked dependency.
     * @throws MojoExecutionException if there is a problem while unpacking.
     */
    private File unpackToTempDirectory(Artifact artifact)
        throws MojoExecutionException
    {
        String name = artifact.getFile().getName();
        File tempLocation = new File(workDirectory, name.substring(0, name.length() - 4));

        boolean process = false;
        if (!tempLocation.exists())
        {
            tempLocation.mkdirs();
            process = true;
        }
        else if (artifact.getFile().lastModified() > tempLocation.lastModified())
        {
            process = true;
        }

        if (process)
        {
            File file = artifact.getFile();
            try
            {
                unpack(file, tempLocation);
            }
            catch (NoSuchArchiverException e)
            {
                this.getLog().info("Skip unpacking dependency file with unknown extension: " + file.getPath());
            }
        }

        return tempLocation;
    }

    /**
     * Unpacks the archive file.
     *
     * @param file     File to be unpacked.
     * @param location Location where to put the unpacked files.
     * @throws org.codehaus.plexus.archiver.manager.NoSuchArchiverException
     *          if no archiver is found.
     * @throws org.apache.maven.plugin.MojoExecutionException
     *          if there is a problem while executing.
     */
    private void unpack(File file, File location)
        throws MojoExecutionException, NoSuchArchiverException
    {
        String archiveExt = FileUtils.getExtension(file.getAbsolutePath()).toLowerCase();

        try
        {
            UnArchiver unArchiver = archiverManager.getUnArchiver(archiveExt);
            unArchiver.setSourceFile(file);
            unArchiver.setDestDirectory(location);
            unArchiver.setOverwrite(true);
            unArchiver.extract();
        }
        catch (ArchiverException e)
        {
            throw new MojoExecutionException("Error unpacking file: " + file + "to: " + location, e);
        }
    }

    /**
     * Recursively copies contents of <tt>srcDir</tt> into <tt>targetDir</tt>.
     * This will not overwrite any existing files.
     *
     * @param srcDir    Directory containing unpacked dependency contents
     * @param targetDir Directory to overlay srcDir into
     * @throws org.apache.maven.plugin.MojoExecutionException
     *          if there is a problem while executing
     */
    private void copyDependencyContents(File srcDir, File targetDir)
        throws MojoExecutionException
    {
        DirectoryScanner scanner = new DirectoryScanner();
        scanner.setBasedir(srcDir);
        scanner.setExcludes(getDependencyExcludes());
        scanner.addDefaultExcludes();

        scanner.setIncludes(getDependencyIncludes());

        scanner.scan();

        String[] dirs = scanner.getIncludedDirectories();
        for (int j = 0; j < dirs.length; j++)
        {
            new File(targetDir, dirs[j]).mkdirs();
        }

        String[] files = scanner.getIncludedFiles();

        for (int j = 0; j < files.length; j++)
        {
            File srcFile = new File(srcDir, files[j]);
            File targetFile = new File(targetDir, files[j]);

            try
            {
                File parentDir = targetFile.getParentFile();
                parentDir.mkdirs();
                if (!parentDir.isDirectory())
                {
                    getLog().warn("Skipping " + files[j] + ": failed to create parent directory " + parentDir);
                }
                else if ( (targetFile.isDirectory() && srcFile.isFile()) ||
                          (srcFile.isDirectory() && targetFile.isFile())
                        )
                {
                    getLog().warn("Skipping " + files[j] + ": conflicts with existing file " + targetFile);
                }
                else
                {
                    copyFileIfModified(srcFile, targetFile);
                }
            }
            catch (IOException e)
            {
                throw new MojoExecutionException("Error copying file '" + files[j] + "' to '" + targetFile + "'", e);
            }
        }
    }

    /**
     * Returns a string array of the excludes to be used
     * when adding dependentcies as an overlay onto this artifact.
     *
     * @return an array of tokens to exclude
     */
    protected String[] getDependencyExcludes()
    {
        String[] excludes;
        if (StringUtils.isNotEmpty(dependencyExcludes))
        {
            excludes = StringUtils.split(dependencyExcludes, ",");
        }
        else
        {
            excludes = EMPTY_STRING_ARRAY;
        }
        return excludes;
    }

    /**
     * Returns a string array of the includes to be used
     * when adding dependencies as an overlay onto this artifact.
     *
     * @return an array of tokens to include
     */
    protected String[] getDependencyIncludes()
    {
        return StringUtils.split(StringUtils.defaultString(dependencyIncludes), ",");
    }

    public void setExtractDependencies(boolean extractDependencies)
    {
        this.extractDependencies = extractDependencies;
    }

    public MavenProject getProject()
    {
        return project;
    }

    public void setProject(MavenProject project)
    {
        this.project = project;
    }

    public String getLocalRepository()
    {
        return localRepository;
    }

    public void setLocalRepository(String localRepository)
    {
        this.localRepository = localRepository;
    }

    public String getTarget()
    {
        return target;
    }

    public void setTarget(String target)
    {
        this.target = target;
    }

    public void setWorkDirectory(File workDirectory)
    {
        this.workDirectory = workDirectory;
    }

    public void setDependencyExcludes(String dependencyExcludes)
    {
        this.dependencyExcludes = dependencyExcludes;
    }
}
