package com.atlassian.maven.plugins.pdk;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.plugin.AbstractMojo;

public abstract class BasePdkMojo extends AbstractMojo {

    /**
     * @parameter expression="${localRepository}"
     */
    protected ArtifactRepository localRepository;

    /**
     * @component
     */
    protected ArtifactFactory artifactFactory;

    protected void copyFiles( File source, File dest ) {

        try {
            if ( source.isDirectory() ) {
                File[] files = source.listFiles();

                // loop in all files and folders found under this folder
                if ( files.length != 0 ) {
                    getLog().info( "Copying " + files.length + " files to " + dest.getPath() );
                }
                for ( int i = 0; i < files.length; i++ ) {
                    // if the File object is a file then get ready to copy that
                    // file
                    if ( !files[i].isDirectory() ) {
                        copy( files[i], new File( dest, files[i].getName() ) );
                    } else {
                        // if the folder is not "CVS" or ".svn" then copy it
                        if ( !( files[i].getName().equals( "CVS" ) ) && !( files[i].getName().equals( ".svn" ) ) ) {
                            File sourceDir = new File( source + File.separator + files[i].getName() );
                            File destDir = new File( dest + File.separator + files[i].getName() );
                            destDir.mkdir();
                            // recursively creates all files under this folder
                            copyFiles( sourceDir, destDir );
                        }
                    }
                }
            } else if ( source.exists() ) {
                copy( source, dest );
            }
        } catch ( IOException e ) {
            getLog().error( e );
        }
    }

    /**
     * Fast & simple file copy.
     * 
     * @param source
     *            The source file.
     * @param dest
     *            The destination file.
     * @throws java.io.IOException
     *             if there is a problem while copying.
     */
    protected static void copy( File source, File dest ) throws IOException {

        if ( source.isDirectory() ) {
            FileChannel in = null, out = null;
            try {
                in = new FileInputStream( source ).getChannel();
                out = new FileOutputStream( dest ).getChannel();

                long size = in.size();
                MappedByteBuffer buf = in.map( FileChannel.MapMode.READ_ONLY, 0, size );

                out.write( buf );
            } catch ( IOException t ) {
                throw t;
            } finally {
                if ( in != null )
                    in.close();
                if ( out != null )
                    out.close();
                dest.setLastModified( source.lastModified() );
            }
        } else {
            if ( !dest.isDirectory() ) {
                dest = new File( dest.getParent() );
            }
            InputStream in = new FileInputStream( source );
            OutputStream out = new FileOutputStream( new File( dest.getPath() + File.separator + source.getName() ) );

            // Transfer bytes from in to out
            byte[] buf = new byte[1024];
            int len;
            while ( ( len = in.read( buf ) ) > 0 ) {
                out.write( buf, 0, len );
            }
            in.close();
            out.close();
        }
    }

    protected Artifact getArtifact( String groupId, String artifactId, String version, String type ) {

        return artifactFactory.createArtifact( groupId, artifactId, version, Artifact.SCOPE_RUNTIME, type );
    }

    protected String urlEncode( String text ) {

        try {
            return URLEncoder.encode( text, "UTF-8" );
        } catch ( UnsupportedEncodingException e ) {
            return text;
        }
    }
}