package com.atlassian.event.internal;

import com.atlassian.event.api.EventListener;
import com.atlassian.event.spi.ListenerHandler;
import com.atlassian.event.spi.ListenerInvoker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Method;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A listener handler that will check for single parameter methods annotated with the given annotation.
 * <p>
 * The default annotation for methods is {@link EventListener}.
 *
 * @see EventListener
 * @since 2.0
 */
public final class AnnotatedMethodsListenerHandler implements ListenerHandler {
    private final Logger log = LoggerFactory.getLogger(this.getClass());

    private final Class annotationClass;

    public AnnotatedMethodsListenerHandler() {
        this(EventListener.class);
    }

    public AnnotatedMethodsListenerHandler(Class annotationClass) {
        this.annotationClass = checkNotNull(annotationClass);
    }

    public List<? extends ListenerInvoker> getInvokers(final Object listener) {
        final Map<Method, Optional<String>> validMethods = getValidMethods(checkNotNull(listener));

        if (validMethods.isEmpty()) {
            log.debug("Couldn't find any valid listener methods on class <{}>", listener.getClass().getName());
        }

        return validMethods.entrySet().stream()
                .map(entry -> new SingleParameterMethodListenerInvoker(listener, entry.getKey(), entry.getValue()))
                .collect(Collectors.toCollection(LinkedList::new));
    }

    private Map<Method, Optional<String>> getValidMethods(Object listener) {
        final Map<Method, Optional<String>> annotatedMethods = new LinkedHashMap<>();
        for (Method method : listener.getClass().getMethods()) {
            if (isValidMethod(method)) {
                annotatedMethods.put(method, getScope(method));
            }
        }
        return annotatedMethods;
    }

    private Optional<String> getScope(Method method) {
        final EventListener annotation = method.getAnnotation(EventListener.class);
        return Optional
                .ofNullable(annotation)
                .map(EventListener::scope)
                .filter(scopeName -> !"".equals(scopeName));
    }

    private boolean isValidMethod(Method method) {
        if (isAnnotated(method)) {
            if (hasOneAndOnlyOneParameter(method)) {
                return true;
            } else {
                throw new RuntimeException("Method <" + method + "> of class <" + method.getDeclaringClass() + "> " +
                        "is annotated with <" + annotationClass.getName() + "> but has 0 or more than 1 parameters! " +
                        "Listener methods MUST have 1 and only 1 parameter.");
            }
        }
        return false;
    }

    @SuppressWarnings("unchecked")
    private boolean isAnnotated(Method method) {
        return method.getAnnotation(annotationClass) != null;
    }

    private boolean hasOneAndOnlyOneParameter(Method method) {
        return method.getParameterTypes().length == 1;
    }
}
