/*
 * Decompiled with CFR 0.152.
 */
package io.snyk.jenkins.tools;

import hudson.Extension;
import hudson.FilePath;
import hudson.Functions;
import hudson.Util;
import hudson.model.Node;
import hudson.model.TaskListener;
import hudson.remoting.Callable;
import hudson.remoting.VirtualChannel;
import hudson.tools.ToolInstallation;
import hudson.tools.ToolInstaller;
import hudson.tools.ToolInstallerDescriptor;
import hudson.util.ListBoxModel;
import io.snyk.jenkins.tools.Platform;
import io.snyk.jenkins.tools.PlatformItem;
import io.snyk.jenkins.tools.SnykInstallation;
import io.snyk.jenkins.tools.internal.DownloadService;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import jenkins.security.MasterToSlaveCallable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.kohsuke.stapler.DataBoundConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SnykInstaller
extends ToolInstaller {
    private static final Logger LOG = LoggerFactory.getLogger((String)SnykInstaller.class.getName());
    private static final String INSTALLED_FROM = ".installedFrom";
    private static final String TIMESTAMP_FILE = ".timestamp";
    private final String version;
    private final Long updatePolicyIntervalHours;
    private final PlatformItem platform;

    @DataBoundConstructor
    public SnykInstaller(String label, String version, Long updatePolicyIntervalHours, PlatformItem platform) {
        super(label);
        this.version = version;
        this.updatePolicyIntervalHours = updatePolicyIntervalHours;
        this.platform = platform;
    }

    public FilePath performInstallation(ToolInstallation tool, Node node, TaskListener listener) throws IOException, InterruptedException {
        FilePath expected = this.preferredLocation(tool, node);
        PrintStream logger = listener.getLogger();
        if (this.isUpToDate(expected)) {
            return expected;
        }
        logger.println("Installing Snyk (" + Util.fixEmptyAndTrim((String)this.version) + ")...");
        return this.downloadSnykBinaries(expected, node);
    }

    private boolean isUpToDate(FilePath expectedLocation) throws IOException, InterruptedException {
        long timestampFromFile;
        FilePath marker = expectedLocation.child(TIMESTAMP_FILE);
        if (!marker.exists()) {
            return false;
        }
        String content = StringUtils.chomp((String)marker.readToString());
        try {
            timestampFromFile = Long.parseLong(content);
        }
        catch (NumberFormatException ex) {
            LOG.error(".timestamp file is corrupt and cannot be read and will be reset to 0.");
            timestampFromFile = 0L;
        }
        long timestampNow = Instant.now().toEpochMilli();
        long timestampDifference = timestampNow - timestampFromFile;
        if (timestampDifference <= 0L) {
            return true;
        }
        long updateInterval = TimeUnit.HOURS.toMillis(this.updatePolicyIntervalHours);
        return timestampDifference < updateInterval;
    }

    private FilePath downloadSnykBinaries(FilePath expected, Node node) {
        for (String snykUrlTemplate : DownloadService.SNYK_CLI_DOWNLOAD_URLS) {
            try {
                LOG.info("Installing Snyk '{}' on Build Node '{}'", (Object)this.version, (Object)node.getDisplayName());
                VirtualChannel nodeChannel = node.getChannel();
                if (nodeChannel == null) {
                    throw new IOException(String.format("Build Node '%s' is offline.", node.getDisplayName()));
                }
                Platform platform = PlatformItem.convert(this.platform);
                if (platform == null) {
                    LOG.info("Installer architecture is not configured or use AUTO mode");
                    platform = (Platform)((Object)nodeChannel.call((Callable)new GetPlatform()));
                }
                LOG.info("Configured installer architecture is {}", (Object)platform);
                URL snykDownloadUrl = DownloadService.constructDownloadUrlForSnyk(snykUrlTemplate, "cli", this.version, platform);
                URL snykToHtmlDownloadUrl = DownloadService.constructDownloadUrlForSnyk(snykUrlTemplate, "snyk-to-html", "latest", platform);
                LOG.info("Downloading CLI from {}", (Object)snykDownloadUrl);
                LOG.info("Downloading snyk-to-html from {}", (Object)snykToHtmlDownloadUrl);
                expected.mkdirs();
                nodeChannel.call((Callable)new Downloader(snykDownloadUrl, expected.child(platform.snykWrapperFileName)));
                nodeChannel.call((Callable)new Downloader(snykToHtmlDownloadUrl, expected.child(platform.snykToHtmlWrapperFileName)));
                expected.child(INSTALLED_FROM).write(snykDownloadUrl.toString(), StandardCharsets.UTF_8.name());
                expected.child(TIMESTAMP_FILE).write(String.valueOf(Instant.now().toEpochMilli()), StandardCharsets.UTF_8.name());
                return expected;
            }
            catch (IOException | InterruptedException | RuntimeException ex) {
                LOG.error("Failed to install Snyk.", (Throwable)ex);
            }
        }
        throw new RuntimeException("Failed to install Snyk.");
    }

    public String getVersion() {
        return this.version;
    }

    public Long getUpdatePolicyIntervalHours() {
        return this.updatePolicyIntervalHours;
    }

    public PlatformItem getPlatform() {
        return this.platform;
    }

    private static class GetPlatform
    extends MasterToSlaveCallable<Platform, IOException> {
        private static final long serialVersionUID = 1L;

        private GetPlatform() {
        }

        public Platform call() {
            return Platform.current();
        }
    }

    private static class Downloader
    extends MasterToSlaveCallable<Void, IOException> {
        private static final long serialVersionUID = 1L;
        private final URL downloadUrl;
        private final FilePath output;

        Downloader(URL downloadUrl, FilePath output) {
            this.downloadUrl = downloadUrl;
            this.output = output;
        }

        public Void call() throws IOException {
            boolean result;
            File downloadedFile = new File(this.output.getRemote());
            FileUtils.copyURLToFile((URL)this.downloadUrl, (File)downloadedFile, (int)10000, (int)10000);
            if (!Functions.isWindows() && downloadedFile.isFile() && !(result = downloadedFile.setExecutable(true, false))) {
                throw new RuntimeException(String.format("Failed to set file as executable. (%s)", downloadedFile.getAbsolutePath()));
            }
            return null;
        }
    }

    @Extension
    public static final class SnykInstallerDescriptor
    extends ToolInstallerDescriptor<SnykInstaller> {
        @Nonnull
        public String getDisplayName() {
            return "Install from snyk.io";
        }

        public boolean isApplicable(Class<? extends ToolInstallation> toolType) {
            return toolType == SnykInstallation.class;
        }

        public ListBoxModel doFillPlatformItems() {
            ListBoxModel platformItems = new ListBoxModel();
            platformItems.add("Auto-detection", PlatformItem.AUTO.name());
            platformItems.add("Linux (amd64)", PlatformItem.LINUX.name());
            platformItems.add("Linux Alpine (amd64)", PlatformItem.LINUX_ALPINE.name());
            platformItems.add("Mac OS (amd64)", PlatformItem.MAC_OS.name());
            platformItems.add("Windows (amd64)", PlatformItem.WINDOWS.name());
            return platformItems;
        }
    }
}

