package io.jenkins.plugins.neuvector;

import com.google.common.base.Strings;
import hudson.AbortException;
import hudson.EnvVars;
import hudson.Extension;
import hudson.FilePath;
import hudson.Launcher;
import hudson.model.AbstractProject;
import hudson.model.Run;
import hudson.model.TaskListener;
import hudson.tasks.ArtifactArchiver;
import hudson.tasks.Builder;
import hudson.tasks.BuildStepDescriptor;
import hudson.util.FormValidation;
import hudson.util.ListBoxModel;
import hudson.util.Secret;

import java.io.*;
import java.net.HttpURLConnection;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.ArrayList;

import jenkins.tasks.SimpleBuildStep;
import net.sf.json.JSONArray;
import net.sf.json.JSONException;
import net.sf.json.JSONObject;
import org.jenkinsci.Symbol;
import org.kohsuke.stapler.DataBoundConstructor;
import org.kohsuke.stapler.DataBoundSetter;
import org.kohsuke.stapler.StaplerRequest;
import org.kohsuke.stapler.QueryParameter;

import javax.annotation.Nonnull;
import io.jenkins.plugins.neuvector.model.ScanMeta;

import java.util.Set;
import java.util.HashSet;

public class NeuVectorBuilder extends Builder implements SimpleBuildStep {

    private static final String REGISTRY_DROPDOWN_DEFAULT = "Please select a registry:";
    private static final String LOCAL_REGIESTRY = "Local";
    private static final String CONTROLLER_ENDPOINT_DROPDOWN_DEFAULT = "Please select a controller endpoint:";
    private boolean standaloneScanner;
    private boolean sendReportToController;
    private final String repository;
    private String tag;
    private boolean scanLayers;
    private String numberOfHighSeverityToFail;
    private String numberOfMediumSeverityToFail;
    private String nameOfVulnerabilityToFailOne;
    private String nameOfVulnerabilityToFailTwo;
    private String nameOfVulnerabilityToFailThree;
    private String nameOfVulnerabilityToFailFour;
    private String nameOfVulnerabilityToExemptOne;
    private String nameOfVulnerabilityToExemptTwo;
    private String nameOfVulnerabilityToExemptThree;
    private String nameOfVulnerabilityToExemptFour;
    private Integer scanTimeout;
    
    // not set to default for backwards compatibility
    private String controllerEndpointUrlSelection;
    private String registrySelection;

    private static Integer buildStep;
    private static Integer buildHash = 0;
    private Log logger;
    
    private boolean isLocal = true;
    private Registry selectRegistry = null;
    private ControllerEndpointUrl selectControllerEndpointUrl = null;

    public synchronized static void setBuildStep(Integer buildStep) {
        NeuVectorBuilder.buildStep = buildStep;
    }

    public synchronized static void setBuildHash(Integer buildHash) {
        NeuVectorBuilder.buildHash = buildHash;
    }

    @DataBoundConstructor
    public NeuVectorBuilder(String repository, String registrySelection) {
        this.repository = repository;

        if ((registrySelection != null) && (!registrySelection.equals(REGISTRY_DROPDOWN_DEFAULT))) {
            this.registrySelection = registrySelection;
        } else {
            this.registrySelection = null;
        }
    }

    public boolean getStandaloneScanner() {return standaloneScanner;}

    public boolean getSendReportToController() {
        return sendReportToController;
    }

    public String getRepository() {
        return repository;
    }

    public String getTag() {
        return tag;
    }

    public boolean getScanLayers() {
        return scanLayers;
    }

    public Integer getScanTimeout() { return scanTimeout; }

    public String getNumberOfHighSeverityToFail() {
        return numberOfHighSeverityToFail;
    }

    public String getNumberOfMediumSeverityToFail() {
        return numberOfMediumSeverityToFail;
    }

    public String getNameOfVulnerabilityToFailOne() {
        return nameOfVulnerabilityToFailOne;
    }

    public String getNameOfVulnerabilityToFailTwo() {
        return nameOfVulnerabilityToFailTwo;
    }

    public String getNameOfVulnerabilityToFailThree() {
        return nameOfVulnerabilityToFailThree;
    }

    public String getNameOfVulnerabilityToFailFour() {
        return nameOfVulnerabilityToFailFour;
    }

    public String getNameOfVulnerabilityToExemptOne() {
        return nameOfVulnerabilityToExemptOne;
    }

    public String getNameOfVulnerabilityToExemptTwo() {
        return nameOfVulnerabilityToExemptTwo;
    }

    public String getNameOfVulnerabilityToExemptThree() {
        return nameOfVulnerabilityToExemptThree;
    }

    public String getNameOfVulnerabilityToExemptFour() {
        return nameOfVulnerabilityToExemptFour;
    }

    public String getRegistrySelection() {
        return registrySelection;
    }

    public String getControllerEndpointUrlSelection() {
        return controllerEndpointUrlSelection;
    }

    @DataBoundSetter
    public void setStandaloneScanner(boolean standaloneScanner) { this.standaloneScanner = standaloneScanner; }
    
    @DataBoundSetter
    public void setSendReportToController(boolean sendReportToController) {
        this.sendReportToController = sendReportToController;
    }

    @DataBoundSetter
    public void setTag(String tag) {
        this.tag = tag.trim();
    }

    @DataBoundSetter
    public void setScanLayers(boolean scanLayers) {
        this.scanLayers = scanLayers;
    }

    @DataBoundSetter
    public void setScanTimeout(Integer scanTimeout) { this.scanTimeout = (scanTimeout == null) ? Integer.valueOf(0) : scanTimeout; }

    @DataBoundSetter
    public void setNumberOfHighSeverityToFail(String numberOfHighSeverityToFail) {
        try {
            Integer.parseInt(numberOfHighSeverityToFail.trim());
            this.numberOfHighSeverityToFail = numberOfHighSeverityToFail.trim();
        } catch (NumberFormatException e) {
            this.numberOfHighSeverityToFail = "";
        }
    }

    @DataBoundSetter
    public void setNumberOfMediumSeverityToFail(String numberOfMediumSeverityToFail) {
        try {
            Integer.parseInt(numberOfMediumSeverityToFail.trim());
            this.numberOfMediumSeverityToFail = numberOfMediumSeverityToFail.trim();
        } catch (NumberFormatException e) {
            this.numberOfMediumSeverityToFail = "";
        }
    }

    @DataBoundSetter
    public void setNameOfVulnerabilityToFailOne(String nameOfVulnerabilityToFailOne) {
        this.nameOfVulnerabilityToFailOne = nameOfVulnerabilityToFailOne.trim();
    }

    @DataBoundSetter
    public void setNameOfVulnerabilityToFailTwo(String nameOfVulnerabilityToFailTwo) {
        this.nameOfVulnerabilityToFailTwo = nameOfVulnerabilityToFailTwo.trim();
    }

    @DataBoundSetter
    public void setNameOfVulnerabilityToFailThree(String nameOfVulnerabilityToFailThree) {
        this.nameOfVulnerabilityToFailThree = nameOfVulnerabilityToFailThree.trim();
    }

    @DataBoundSetter
    public void setNameOfVulnerabilityToFailFour(String nameOfVulnerabilityToFailFour) {
        this.nameOfVulnerabilityToFailFour = nameOfVulnerabilityToFailFour.trim();
    }

    @DataBoundSetter
    public void setNameOfVulnerabilityToExemptOne(String nameOfVulnerabilityToExemptOne) {
        this.nameOfVulnerabilityToExemptOne = nameOfVulnerabilityToExemptOne.trim();
    }

    @DataBoundSetter
    public void setNameOfVulnerabilityToExemptTwo(String nameOfVulnerabilityToExemptTwo) {
        this.nameOfVulnerabilityToExemptTwo = nameOfVulnerabilityToExemptTwo.trim();
    }

    @DataBoundSetter
    public void setNameOfVulnerabilityToExemptThree(String nameOfVulnerabilityToExemptThree) {
        this.nameOfVulnerabilityToExemptThree = nameOfVulnerabilityToExemptThree.trim();
    }

    @DataBoundSetter
    public void setNameOfVulnerabilityToExemptFour(String nameOfVulnerabilityToExemptFour) {
        this.nameOfVulnerabilityToExemptFour = nameOfVulnerabilityToExemptFour.trim();
    }

    @DataBoundSetter
    public void setControllerEndpointUrlSelection(String controllerEndpointUrlSelection) {
        if ((controllerEndpointUrlSelection != null) && (!controllerEndpointUrlSelection.equals(CONTROLLER_ENDPOINT_DROPDOWN_DEFAULT))) {
            this.controllerEndpointUrlSelection = controllerEndpointUrlSelection;
        } else {
            this.controllerEndpointUrlSelection = null;
        }
    }


    @Override
    public void perform(@Nonnull Run<?, ?> run, @Nonnull FilePath workspace, @Nonnull Launcher launcher, @Nonnull TaskListener listener)
            throws IOException, InterruptedException {

        // init the logger
        this.logger = new Log(listener.getLogger());

        // copy styles.css to workspace
        File cssFile;
        final EnvVars env = run.getEnvironment(listener);
        FilePath targetCss = new FilePath(workspace, "styles.css");

        cssFile = new File(env.get("JENKINS_HOME") + "/plugins/neuvector-vulnerability-scanner/css/", "styles.css");
        FilePath cssFilePath = new FilePath(cssFile);
        cssFilePath.copyTo(targetCss);

        String artifactName = "NeuVectorReport_" + run.getParent().getDisplayName() + "_" + run.getNumber();
        String reportNumber;
        //get the build step number as the part of the artifact name
        if (run.hashCode() != buildHash) {
            setBuildHash(run.hashCode());
            setBuildStep(1);
        } else {
            setBuildStep(buildStep + 1);
        }
        reportNumber = Integer.toString(buildStep);
        artifactName = artifactName + "-" + reportNumber ;

        //to init config with scan parameters
        Config config = null;
        try {
            config = printInfoFromUser(run, listener);
        } catch (URISyntaxException e) {
            e.printStackTrace();
        }

        //to create a worker object to run the scan
        NeuVectorWorker worker = new NeuVectorWorker(logger, config, workspace, artifactName);
        try{
            worker.scan(run, launcher);
        }catch (Exception ex) {
            archiveAndAddAction(run, workspace, launcher, listener, artifactName, reportNumber);
            throw new AbortException(ex.getMessage());
        }

        archiveAndAddAction(run, workspace, launcher, listener, artifactName, reportNumber);

    }

    private String getBuildUser(String url) throws IOException{
        StringBuilder result = new StringBuilder();
        URL urlVar = new URL(url);
        HttpURLConnection conn = (HttpURLConnection) urlVar.openConnection();
        conn.setRequestMethod("GET");
        try(BufferedReader rd = new BufferedReader(new InputStreamReader(conn.getInputStream(), StandardCharsets.UTF_8))){
            String line;
            while ((line = rd.readLine()) != null) {
                result.append(line);
            }
        }

        String userName = "";
        try{
            JSONObject json = JSONObject.fromObject(result.toString());
            JSONObject action = (JSONObject) json.getJSONArray("actions").get(0);
            JSONObject cause = (JSONObject) action.getJSONArray("causes").get(0);
            userName = cause.getString("userName");
        }catch(JSONException e){
            userName = "anonymous";
        }

        return userName;
    }

    private void selectRegistryInDropDown(List<Registry> registries) throws AbortException{
        if (registrySelection.equalsIgnoreCase(REGISTRY_DROPDOWN_DEFAULT)) {
            // Case: no registry selected in task.
            throw new AbortException("Please select a registry or choose " + LOCAL_REGIESTRY);
        } else if (registrySelection.equalsIgnoreCase(LOCAL_REGIESTRY)) {
            // Case: Local.
            logger.println("Local case.");
        } else {
            // Case: pre-defined registry selected. Maybe deleted already.
            if (registries == null) {
                // Case: registry deleted in global
                throw new AbortException("Registry " + registrySelection + " in current project is missing or deleted in global configuration.");
            } else {
                boolean found = false;
                for (Registry registry : registries) {
                    if (registry.getNickname().isEmpty() || registry.getRegUrl().isEmpty()) {
                        throw new AbortException("Registry nickname and URL cannot be empty");
                    }
                    if (registry.getNickname().equalsIgnoreCase(registrySelection)) {
                        // Case: registry found in global
                        found = true;
                        isLocal = false;
                        selectRegistry = registry;
                        logger.println("Registry: " + registrySelection);
                        logger.println("Registry URL: " + selectRegistry.getRegUrl());
                        logger.println("Registry Username: " + selectRegistry.getRegUsername());
                        break;
                    }
                }
                if (!found) {
                    // Case: registry deleted in global
                    throw new AbortException("Registry " + registrySelection + " in current project is missing or deleted in global configuration.");
                }
            }            
        }
    }
    
    private void selectControllerEndpointUrlInDropDown(List<ControllerEndpointUrl> controllerEndpointUrls) throws AbortException{
        if (Strings.isNullOrEmpty(controllerEndpointUrlSelection)) {
            // Case: no controller endpoint URL selection provided, treat as optional
            logger.println("No controller endpoint URL selection provided, proceeding without it.");
            return;
        }

        if (controllerEndpointUrlSelection.equalsIgnoreCase(CONTROLLER_ENDPOINT_DROPDOWN_DEFAULT)) {
            // Case: no registry selected in task.
            throw new AbortException("Please select a Controller Endpoint.");
        } else {
            // Case: pre-defined registry selected. Maybe deleted already.
            if (controllerEndpointUrls == null) {
                // Case: registry deleted in global
                throw new AbortException("Controller Endpoint " + controllerEndpointUrlSelection + " in current project is missing or deleted in global configuration.");
            } else {
                boolean found = false;
                for (ControllerEndpointUrl controllerEndpointUrl : controllerEndpointUrls) {
                    if (controllerEndpointUrl.getNickname().isEmpty() || controllerEndpointUrl.getControllerEndpointUrl().isEmpty()) {
                        throw new AbortException("Controller Endpoint nickname and URL cannot be empty");
                    }

                    if (controllerEndpointUrl.getNickname().equalsIgnoreCase(controllerEndpointUrlSelection)) {
                        // Case: registry found in global
                        found = true;
                        isLocal = false;
                        selectControllerEndpointUrl = controllerEndpointUrl;
                        logger.println("Controller Endpoint : " + controllerEndpointUrlSelection);
                        logger.println("Controller Endpoint URL: " + selectControllerEndpointUrl.getControllerEndpointUrl());
                        logger.println("Controller Endpoint Username: " + selectControllerEndpointUrl.getControllerUser());
                        break;
                    }
                }
                if (!found) {
                    // Case: Controller Endpoint deleted in global
                    throw new AbortException("Controller Endpoint " + controllerEndpointUrlSelection + " in current project is missing or deleted in global configuration.");
                }
            }            
        }
    } 

    private Config printInfoFromUser(Run<?, ?> run, TaskListener listener) throws AbortException, URISyntaxException {
        DescriptorImpl globalConfig = getDescriptor();

        List<Registry> registries = globalConfig.getRegistries(); // registries can be null if nothing in global config
        List<ControllerEndpointUrl> controllerEndpointUrls = globalConfig.getControllerEndpointUrls(); // controllerEndpointUrls can be null if nothing in global config
        if (repository == null || registrySelection == null) {
            throw new AbortException("repository and registrySelection both are required.");
        }

        if (controllerEndpointUrlSelection == null && globalConfig.getIsDeafultControllerEndpointUrls()) {
            controllerEndpointUrlSelection = "default";
        }
        
        final EnvVars env;
        try {
            env = run.getEnvironment(listener);
        } catch (IOException | InterruptedException e) {
            throw new AbortException("Error when getting Jenkins project environment.");
        }
        String currentRepository = env.expand(repository.trim());
        String currentTag = env.expand(tag);

        String buildUrl = env.get("BUILD_URL")+"api/json";
        String buildUser = "";
        try {
            buildUser = getBuildUser(buildUrl);
        } catch (IOException e) {
            logger.println("Build user not found.");
        }

        logger.println("");
        logger.println("*************************************************************");

        ScanMeta scanMeta = new ScanMeta();
        scanMeta.setSource(globalConfig.getSource());
        scanMeta.setUser(buildUser);
        scanMeta.setJob(env.get("JOB_NAME"));
        scanMeta.setWorkspace(env.get("WORKSPACE"));

        if(scanMeta.getSource() == null || scanMeta.getSource().isEmpty()){
            logger.println("Scanner Source Name: (Please set a name in Jenkins -> Configure System -> NeuVector Vulnerablity Scanner -> NeuVector Scanner Source Name )");
        }else{
            logger.println("Scanner Source Name: " + scanMeta.getSource());
        }

        logger.println("Build User: " + scanMeta.getUser());
        logger.println("Job Name: " + scanMeta.getJob());
        logger.println("Workspace: " + scanMeta.getWorkspace());

        selectRegistryInDropDown(registries);
        selectControllerEndpointUrlInDropDown(controllerEndpointUrls);

        logger.println("Repository: " + currentRepository);
        logger.println("Tag: " + currentTag);

        Double highSeverityThreshold = globalConfig.highSeverityThreshold;
        Double mediumSeverityThreshold = globalConfig.mediumSeverityThreshold;
        boolean customizedRatingScale = false;
        if(highSeverityThreshold != null && mediumSeverityThreshold != null && (highSeverityThreshold > mediumSeverityThreshold)) {
            customizedRatingScale = true;
        }

        return new Config.ConfigBuilder(selectControllerEndpointUrl)
                .standaloneScanner(standaloneScanner)
                .sendReportToController(sendReportToController)
                .scanMeta(scanMeta)
                .isLocal(isLocal)
                .scannerRegistryURL(globalConfig.scannerRegistryURL)
                .scannerRegistryUser(globalConfig.scannerRegistryUser)
                .scannerRegistryPassword(globalConfig.scannerRegistryPassword)
                .scannerImage(globalConfig.scannerImage)
                .registry(selectRegistry)
                .repository(currentRepository)
                .tag(currentTag)
                .scanLayers(scanLayers)
                .scanTimeout(scanTimeout)
                .highSeverityThreshold(highSeverityThreshold)
                .mediumSeverityThreshold(mediumSeverityThreshold)
                .customizedRatingScale(customizedRatingScale)
                .numberOfHighSeverityToFail(numberOfHighSeverityToFail)
                .numberOfMediumSeverityToFail(numberOfMediumSeverityToFail)
                .nameOfVulnerabilityToFailOne(nameOfVulnerabilityToFailOne)
                .nameOfVulnerabilityToFailTwo(nameOfVulnerabilityToFailTwo)
                .nameOfVulnerabilityToFailThree(nameOfVulnerabilityToFailThree)
                .nameOfVulnerabilityToFailFour(nameOfVulnerabilityToFailFour)
                .nameOfVulnerabilityToExemptOne(nameOfVulnerabilityToExemptOne)
                .nameOfVulnerabilityToExemptTwo(nameOfVulnerabilityToExemptTwo)
                .nameOfVulnerabilityToExemptThree(nameOfVulnerabilityToExemptThree)
                .nameOfVulnerabilityToExemptFour(nameOfVulnerabilityToExemptFour)
                .build();
    }

    private void archiveAndAddAction(Run<?, ?> run, FilePath workspace, Launcher launcher, TaskListener listener, String artifactName, String reportNumber){

        try {
            ArtifactArchiver artifactArchiver = new ArtifactArchiver(artifactName);
            artifactArchiver.perform(run, workspace, launcher, listener);
        } catch (Exception e) {
            logger.println("Failed to archive the artifact file: " + artifactName);
        }

        String artifactNameHTML = artifactName + ".html";
        try {
            ArtifactArchiver artifactArchiver = new ArtifactArchiver(artifactNameHTML);
            artifactArchiver.perform(run, workspace, launcher, listener);
        } catch (Exception e) {
            logger.println("Failed to archive the artifact HTML: " + artifactNameHTML);
        }

        String artifactNameJson = artifactName + ".json";
        try {
            ArtifactArchiver artifactArchiver = new ArtifactArchiver(artifactNameJson);
            artifactArchiver.perform(run, workspace, launcher, listener);
        } catch (Exception e) {
            logger.println("Failed to archive the artifact Json: " + artifactNameJson);
        }

        try {
            ArtifactArchiver artifactArchiver = new ArtifactArchiver("styles.css");
            artifactArchiver.perform(run, workspace, launcher, listener);
        } catch (Exception e) {
            logger.println("Failed to archive the artifact css style file. ");
        }

        run.addAction(new NeuVectorAction(run, artifactNameHTML, reportNumber));
    }

    @Override
    public DescriptorImpl getDescriptor() {
        return (DescriptorImpl) super.getDescriptor();
    }

    @Symbol("neuvector")
    @Extension
    public static final class DescriptorImpl extends BuildStepDescriptor<Builder> {
        /**
         * To persist global configuration information, simply store it in a field and
         * call save().
         */
        // Keep these variables for backwards compatibility
        private String source;
        private String controllerIP;
        private String controllerPort;
        private String controllerApiUrl;
        private String user;
        private Secret password;
        private Integer timeout;
        private Boolean disableTLSCertVerification;
        private Boolean disableAPIKeyVerification;
        private String serverCertificate;
        private Secret serverApiKey;

        private String scannerRegistryURL;
        private String scannerImage;
        private String scannerRegistryUser;
        private Secret scannerRegistryPassword;
        private Double mediumSeverityThreshold;
        private Double highSeverityThreshold;
        private List<ControllerEndpointUrl> controllerEndpointUrls;
        private List<Registry> registries;
        private Boolean isDeafultControllerEndpointUrls = false;

        private Double thresholdMin = 0.0;
        private Double thresholdMax = 10.0;

        public void setSource(String source) { this.source = source; }
        public void setScannerRegistryURL(String scannerRegistryURL) {
            this.scannerRegistryURL = scannerRegistryURL;
        }

        public void setScannerImage(String scannerImage) {
            this.scannerImage = scannerImage;
        }
        public void setDisableTLSCertVerification(Boolean disableTLSCertVerification) {this.disableTLSCertVerification = disableTLSCertVerification;}
        public void setDisableAPIKeyVerification(Boolean disableAPIKeyVerification) {this.disableAPIKeyVerification = disableAPIKeyVerification;}
        public void setServerCertificate(String serverCertificate) {this.serverCertificate = serverCertificate;}

        public void setServerApiKey(Secret serverApiKey) {
            this.serverApiKey = serverApiKey;
        }
        
        public void setScannerRegistryUser(String scannerRegistryUser) {
            this.scannerRegistryUser = scannerRegistryUser;
        }

        public void setScannerRegistryPassword(Secret scannerRegistryPassword) {
            this.scannerRegistryPassword = scannerRegistryPassword;
        }

        public void setControllerIP(String controllerIP) {
            this.controllerIP = controllerIP;
        }

        public void setControllerPort(String controllerPort) {
            this.controllerPort = controllerPort;
        }

        public void setControllerApiUrl(String controllerApiUrl) {
            this.controllerApiUrl = controllerApiUrl;
        }

        public void setUser(String user) {
            this.user = user;
        }

        public void setPassword(Secret password) {
            this.password = password;
        }

        public void setTimeout(Integer timeout) {
            this.timeout = timeout;
        }

        public void setRegistries(List<Registry> registries) {
            this.registries = registries;
        }

        public void setControllerEndpointUrls(List<ControllerEndpointUrl> controllerEndpointUrls) {
            this.controllerEndpointUrls = controllerEndpointUrls;
        }

        public void setMediumSeverityThreshold(Double mediumSeverityThreshold) {
            this.mediumSeverityThreshold = mediumSeverityThreshold;
        }

        public void setHighSeverityThreshold(Double highSeverityThreshold) {
            this.highSeverityThreshold = highSeverityThreshold;
        }

        public Double getMediumSeverityThreshold() {
            return mediumSeverityThreshold;
        }

        public Double getHighSeverityThreshold() {
            return highSeverityThreshold;
        }

        public String getSource() {
            return source;
        }

        public String getControllerApiUrl() {
            return controllerApiUrl;
        }

        public String getUser() {
            return user;
        }

        public Secret getPassword() {
            return password;
        }

        public Integer getTimeout() {
            return timeout;
        }

        public String getScannerRegistryURL() {
            return scannerRegistryURL;
        }

        public String getScannerImage() {
            return scannerImage;
        }

        public Boolean getDisableTLSCertVerification() {
            return disableTLSCertVerification;
        }

        public Boolean getDisableAPIKeyVerification() {
            return disableAPIKeyVerification;
        }

        public String getServerCertificate() {
            return serverCertificate;
        }

        public Secret getServerApiKey() {
            return serverApiKey;
        }

        public String getScannerRegistryUser() {
            return scannerRegistryUser;
        }

        public Secret getScannerRegistryPassword() {
            return scannerRegistryPassword;
        }

        public List<Registry> getRegistries() {
            return registries;
        }

        public List<ControllerEndpointUrl> getControllerEndpointUrls() {
            return controllerEndpointUrls;
        }

        public Boolean getIsDeafultControllerEndpointUrls() {
            return isDeafultControllerEndpointUrls;
        }

        public DescriptorImpl() {
            load();
        }

        public ListBoxModel doFillRegistrySelectionItems() {
            ListBoxModel items = new ListBoxModel();
            items.add(REGISTRY_DROPDOWN_DEFAULT);
            items.add(LOCAL_REGIESTRY);
            if (registries != null) {
                for (Registry reg : registries) {
                    items.add(reg.getNickname());
                }
            }
            return items;
        }

        public ListBoxModel doFillControllerEndpointUrlSelectionItems() {
            ListBoxModel items = new ListBoxModel();
            items.add(CONTROLLER_ENDPOINT_DROPDOWN_DEFAULT);
            if (controllerEndpointUrls != null) {
                for (ControllerEndpointUrl controllerEndpointUrl : controllerEndpointUrls) {
                    items.add(controllerEndpointUrl.getNickname());
                }
            }
            return items;
        }

        public FormValidation doCheckScanTimeout(@QueryParameter("scanTimeout") Integer scanTimeout){
            if(scanTimeout > 0 && scanTimeout <=60){
                return FormValidation.ok();
            }
            return FormValidation.error(Messages.NeuVectorBuilder_DescriptorImpl_errors_scanTimeout());
        }

        public FormValidation doCheckRegistrySelection(@QueryParameter String value) {
            if (Strings.isNullOrEmpty(value) || value.equalsIgnoreCase(REGISTRY_DROPDOWN_DEFAULT)) {
                return FormValidation.error(Messages.NeuVectorBuilder_DescriptorImpl_errors_registrySelection());
            }
            return FormValidation.ok();
        }

        public FormValidation doCheckControllerEndpointUrlSelection(@QueryParameter String value) {
            if (Strings.isNullOrEmpty(value) || value.equalsIgnoreCase(CONTROLLER_ENDPOINT_DROPDOWN_DEFAULT)) {
                return FormValidation.error(Messages.NeuVectorBuilder_DescriptorImpl_errors_controllerEndpointUrlSelection());
            }
            return FormValidation.ok();
        }

        public FormValidation doCheckMediumSeverityThreshold(@QueryParameter Double value, @QueryParameter Double highSeverityThreshold) {
            if( value != null){
                if(value < thresholdMin || value > thresholdMax){
                    return FormValidation.error(Messages.NeuVectorBuilder_DescriptorImpl_errors_invalidScore());
                }else if( highSeverityThreshold == null ){
                    return FormValidation.error(Messages.NeuVectorBuilder_DescriptorImpl_errors_missingSeverityScore());
                }
            }

            return FormValidation.ok();
        }

        public FormValidation doCheckHighSeverityThreshold(@QueryParameter Double value, @QueryParameter Double mediumSeverityThreshold) {
            if( value != null){
                if(value < thresholdMin || value > thresholdMax){
                    return FormValidation.error(Messages.NeuVectorBuilder_DescriptorImpl_errors_invalidScore());
                }else if( mediumSeverityThreshold == null ){
                    return FormValidation.error(Messages.NeuVectorBuilder_DescriptorImpl_errors_missingSeverityScore());
                }else if( value <= mediumSeverityThreshold){
                    return FormValidation.error(Messages.NeuVectorBuilder_DescriptorImpl_errors_invalidThreshold());
                }
            }

            return FormValidation.ok();
        }

        private Boolean inRange(Double min, Double max, Double value) {
            if (value != null && value > min && value < max) {
                return true;
            }
            return false;
        }

        private Boolean validThresholds(Double med, Double high) {
            return (inRange(thresholdMin, thresholdMax, med) && inRange(thresholdMin, thresholdMax, high) && (med < high));
        }

        @Override
        public boolean isApplicable(Class<? extends AbstractProject> aClass) {
            return true;
        }

        @Override
        public String getDisplayName() {
            return Messages.NeuVectorBuilder_DescriptorImpl_DisplayName();
        }

        @Override
        public void load() {
            super.load();
        }

        @Override
        public boolean configure(StaplerRequest req, JSONObject formData) {
            formData.put("controllerEndpointUrls", ensureJsonArrayHasNoDuplicateNickname(formData.get("controllerEndpointUrls")));
            formData.put("registries", ensureJsonArrayHasNoDuplicateNickname(formData.get("registries")));
            if (!validThresholds(mediumSeverityThreshold, highSeverityThreshold)) {
                formData.put("highSeverityThreshold", "");
                formData.put("mediumSeverityThreshold", "");
            }
            req.bindJSON(this, formData);
            save();
            return true;
        }

        private JSONArray ensureJsonArrayHasNoDuplicateNickname(Object data) {
            JSONArray jsonArray = new JSONArray();
            if (data instanceof JSONObject) {
                jsonArray.add(data);
            } else if (data instanceof JSONArray) {
                jsonArray = (JSONArray) data;
            }

            JSONArray filteredJsonArray = new JSONArray();
            Set<String> nicknames = new HashSet<>();

            for (int i = 0; i < jsonArray.size(); i++) {
                JSONObject jsonObject = jsonArray.getJSONObject(i);
                String nickname = jsonObject.getString("nickname");
                if (nicknames.add(nickname)) {
                    filteredJsonArray.add(jsonArray.get(i));
                }
            }
            return filteredJsonArray;
        }

        protected Object readResolve() {
            /*
            * This code handles the scenario where users are upgrading from an older version of the plugin,
            * which used controllerIP and port for the scanner API URL. It ensures that the controller IP
            * and port settings are automatically migrated to the new input field "controllerAPIUrl"
            *
            */
            if(controllerApiUrl == null || controllerApiUrl.isEmpty()){
                if(controllerIP != null && !controllerIP.isEmpty() && controllerPort != null && !controllerPort.isEmpty()) {
                    controllerApiUrl = "https://" + controllerIP.trim() + ":" + controllerPort.trim();
                }
            }

            /*
            * TLS certificate verification is enabled by default, but it will be disabled in the following scenario:
            * 
            * To ensure compatibility with users who are upgrading from non-TLS verification version, 
            * we check if the user has previously used API mode when upgrading from non-TLS verification version. 
            * 
            * To check if this is a migrated from non-TLS verification version, we can check serverCertificate is null,
            * since we add serverCertificate when we have TLS verification.
            * 
            * If they have not used Controller & Scanner mode before,
            * verification will remain enabled; otherwise, it will be disabled.
            */

            if (controllerEndpointUrls == null) {
                if (controllerApiUrl != null && !controllerApiUrl.isEmpty()) {
                    controllerEndpointUrls = new ArrayList<>();
                    String defaultNickname = "default";
                    try {
                        boolean migratedFromNonTLSVersion = (serverCertificate == null);
                        if (migratedFromNonTLSVersion) {
                            disableTLSCertVerification = (user != null && !user.isEmpty());
                        }
                        disableAPIKeyVerification = (serverApiKey == null);
                        controllerEndpointUrls.add(new ControllerEndpointUrl(defaultNickname, controllerApiUrl, user, password, disableTLSCertVerification, disableAPIKeyVerification, serverCertificate, timeout, serverApiKey));
                    } catch (URISyntaxException e) {
                        e.printStackTrace();
                    }
                    isDeafultControllerEndpointUrls = true;
                }
            } else {
                // Case Upgrade from 2.4, make sure disableAPIKeyVerification are set as true if user has not configured it
                for (ControllerEndpointUrl controllerEndpointUrl : controllerEndpointUrls) {
                    // if user has not configured disableAPIKeyVerification, set it as true
                    if (controllerEndpointUrl.getDisableAPIKeyVerification() == null) {
                        controllerEndpointUrl.setDisableAPIKeyVerification(true);
                    }
                }
            }

            return this;
          }
    }
}
