package io.jenkins.plugins.neuvector;

import org.apache.http.conn.ssl.TrustStrategy;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;

import java.io.IOException;
import java.io.StringWriter;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

/**
 * A trust strategy that accepts self-signed certificates as trusted. Verification of all other
 * certificates is done by the trust manager configured in the SSL context.
 *
 * @since 4.1
 */
public class BouncyCastleTrustStrategy implements TrustStrategy {

    public static final org.apache.http.conn.ssl.TrustSelfSignedStrategy INSTANCE = new org.apache.http.conn.ssl.TrustSelfSignedStrategy();
    private final String preDefineCert;

    public BouncyCastleTrustStrategy(String userCert) {
        this.preDefineCert = userCert;
    }

    private String normalizeTheCert(String cert) {
        String removePattern = "[\n\\s]+";
        return cert.replaceAll(removePattern, "");
    }

    @Override
    public boolean isTrusted(
            final X509Certificate[] chain, final String authType) throws CertificateException {
        if (chain == null || chain.length == 0) {
            throw new IllegalArgumentException("Certificate chain is null or empty");
        }

        if (chain.length > 1) {
            throw new IllegalArgumentException("Certificate chain should be length one");
        }

        X509Certificate cert = chain[0];
        cert.checkValidity();

        StringWriter sw = new StringWriter();
        JcaPEMWriter pemWriter = new JcaPEMWriter(sw);

        try {
            pemWriter.writeObject(cert);
            pemWriter.close();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        String certFromServer = sw.toString();

        return normalizeTheCert(preDefineCert).equals(normalizeTheCert(certFromServer));
    }

}
