package com.sysdig.jenkins.plugins.sysdig.domain.vm.scanresult;

import com.sysdig.jenkins.plugins.sysdig.domain.AggregateChild;
import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;
import javax.annotation.Nullable;

public class Vulnerability implements AggregateChild<ScanResult>, Serializable {
    private final String cve;
    private final Severity severity;
    private final Date disclosureDate;
    private final Date solutionDate;
    private final boolean exploitable;
    private final String fixVersion;
    private final Set<Package> foundInPackages;
    private final Set<AcceptedRisk> acceptedRisks;
    private final ScanResult root;

    Vulnerability(
            String cve,
            Severity severity,
            Date disclosureDate,
            Date solutionDate,
            boolean exploitable,
            @Nullable String fixVersion,
            ScanResult root) {
        this.cve = cve;
        this.severity = severity;
        this.disclosureDate = disclosureDate;
        this.solutionDate = solutionDate;
        this.exploitable = exploitable;
        this.fixVersion = fixVersion;
        this.foundInPackages = new HashSet<>();
        this.acceptedRisks = new HashSet<>();
        this.root = root;
    }

    public String cve() {
        return cve;
    }

    public Severity severity() {
        return severity;
    }

    public Date disclosureDate() {
        return disclosureDate;
    }

    public Optional<Date> solutionDate() {
        return Optional.ofNullable(solutionDate);
    }

    public boolean exploitable() {
        return exploitable;
    }

    public boolean fixable() {
        return fixVersion().isPresent();
    }

    public Optional<String> fixVersion() {
        return Optional.ofNullable(fixVersion);
    }

    public void addFoundInPackage(Package aPackage) {
        if (this.foundInPackages.add(aPackage)) {
            aPackage.addVulnerabilityFound(this);
        }
    }

    public Set<Package> foundInPackages() {
        return Collections.unmodifiableSet(foundInPackages);
    }

    public Set<Layer> foundInLayers() {
        return foundInPackages.stream().map(Package::foundInLayer).collect(Collectors.toUnmodifiableSet());
    }

    @Override
    public ScanResult root() {
        return root;
    }

    @Override
    public boolean equals(Object o) {
        if (o == null || getClass() != o.getClass()) return false;
        Vulnerability that = (Vulnerability) o;
        return Objects.equals(cve, that.cve);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(cve);
    }

    public void addAcceptedRisk(AcceptedRisk acceptedRisk) {
        if (acceptedRisks.add(acceptedRisk)) {
            acceptedRisk.addForVulnerability(this);
        }
    }

    public Set<AcceptedRisk> acceptedRisks() {
        return Collections.unmodifiableSet(acceptedRisks);
    }
}
