package hudson.plugins.jira.selector;

import static hudson.Util.fixNull;

import com.atlassian.jira.rest.client.api.RestClientException;
import com.atlassian.jira.rest.client.api.domain.Issue;
import hudson.Extension;
import hudson.model.Descriptor;
import hudson.model.Run;
import hudson.model.TaskListener;
import hudson.plugins.jira.EnvironmentExpander;
import hudson.plugins.jira.JiraSession;
import hudson.plugins.jira.JiraSite;
import hudson.plugins.jira.Messages;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.jenkinsci.Symbol;
import org.kohsuke.stapler.DataBoundConstructor;

public class JqlIssueSelector extends AbstractIssueSelector {

    private String jql;

    @DataBoundConstructor
    public JqlIssueSelector(String jql) {
        super();
        this.jql = jql;
    }

    public void setJql(String jql) {
        this.jql = jql;
    }

    public String getJql() {
        return jql;
    }

    @Override
    public Set<String> findIssueIds(Run<?, ?> run, JiraSite site, TaskListener listener) {
        try {
            JiraSession session = site.getSession(run.getParent());
            if (session == null) {
                throw new IllegalStateException("Remote access for Jira isn't configured in Jenkins");
            }

            String expandedJql = EnvironmentExpander.expandVariable(jql, run, listener);

            List<Issue> issues = session.getIssuesFromJqlSearch(expandedJql);

            List<String> issueKeys = new ArrayList<>();

            for (Issue issue : fixNull(issues)) {
                issueKeys.add(issue.getKey());
            }

            // deduplication
            return new HashSet<>(issueKeys);
        } catch (RestClientException e) {
            throw new IllegalStateException("Can't open rest session to Jira site " + site, e);
        }
    }

    @Extension
    @Symbol("JqlSelector")
    public static final class DescriptorImpl extends Descriptor<AbstractIssueSelector> {

        @Override
        public String getDisplayName() {
            return Messages.IssueSelector_JqlIssueSelector_DisplayName();
        }
    }
}
