/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.logic;


import java.io.IOException;
import java.io.InputStream;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.script.ScriptContext;
import javax.script.ScriptException;

import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.resource.Resource;
import net.shibboleth.shared.scripting.AbstractScriptEvaluator;
import net.shibboleth.shared.scripting.EvaluableScript;

import org.slf4j.Logger;

/**
 * A {@link Predicate} which calls out to a supplied script.
 * 
 * @param <T> input type
 * 
 * @since 7.4.0
 */
public class ScriptedPredicate<T> extends AbstractScriptEvaluator implements Predicate<T> {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(ScriptedPredicate.class);

    /** Input type. */
    @Nullable private Class<T> inputTypeClass;

    /**
     * Constructor.
     * 
     * @param theScript the script we will evaluate.
     * @param extraInfo debugging information.
     */
    protected ScriptedPredicate(@Nonnull @NotEmpty final EvaluableScript theScript,
            @Nullable @NotEmpty final String extraInfo) {
        super(theScript);
        setOutputType(Boolean.class);
        setReturnOnError(false);
        setLogPrefix("Scripted Predicate from " + extraInfo + ":");
    }

    /**
     * Constructor.
     * 
     * @param theScript the script we will evaluate.
     */
    protected ScriptedPredicate(@Nonnull @NotEmpty final EvaluableScript theScript) {
        super(theScript);
        setLogPrefix("Anonymous Scripted Predicate:");
        setOutputType(Boolean.class);
        setReturnOnError(false);
    }
    
    /**
     * Get the input type to be enforced.
     *
     * @return input type
     * 
     * @since 8.2.0
     */
    @Nullable public Class<T> getInputType() {
        return inputTypeClass;
    }

    /**
     * Set the input type to be enforced.
     *
     * @param type input type
     * 
     * @since 8.2.0
     */
    public void setInputType(@Nullable final Class<T> type) {
        inputTypeClass = type;
    }
    
    /**
     * Set value to return if an error occurs.
     * 
     * @param flag value to return
     */
    public void setReturnOnError(final boolean flag) {
        setReturnOnError(Boolean.valueOf(flag));
    }
    
    /** {@inheritDoc} */
    public boolean test(@Nullable final T input) {
        
        final Class<T> itype = getInputType();
        if (null != itype && null != input && !itype.isInstance(input)) {
            log.error("{} Input of type {} was not of type {}", getLogPrefix(), input.getClass(),
                    getInputType());
            return returnError();
        }

        final Object result = evaluate(input);
        return (boolean) (result != null ? result : returnError());
    }
    
    /**
     * Helper function to sanity check return-on-error object.
     * 
     * @return a boolean-valued error fallback
     * 
     * @throws ClassCastException if the installed fallback is null or non-Boolean
     */
    private boolean returnError() throws ClassCastException {
        final Object ret = getReturnOnError();
        if (ret instanceof Boolean) {
            return (boolean) ret;
        }
        
        throw new ClassCastException("Unable to cast return value to a boolean");
    }
    
    /** {@inheritDoc} */
    @Override
    protected void prepareContext(@Nonnull final ScriptContext scriptContext, @Nullable final Object... input) {
        scriptContext.setAttribute("input", input != null ? input[0] : null, ScriptContext.ENGINE_SCOPE);
    }
    
    /**
     * Factory to create {@link ScriptedPredicate} from a {@link Resource}.
     * 
     * @param <T> input type
     * @param resource the resource to look at
     * @param engineName the language
     * 
     * @return the predicate
     * 
     * @throws ScriptException if the compile fails
     * @throws IOException if the file doesn't exist.
     * @throws ComponentInitializationException if the scripting initialization fails
     */
    @Nonnull public static <T> ScriptedPredicate<T> resourceScript(
            @Nonnull @NotEmpty @ParameterName(name="engineName") final String engineName,
            @Nonnull @ParameterName(name="resource") final Resource resource)
                    throws ScriptException, IOException, ComponentInitializationException {
        try (final InputStream is = resource.getInputStream()) {
            final EvaluableScript script = new EvaluableScript();
            script.setEngineName(engineName);
            script.setScript(is);
            script.initialize();
            return new ScriptedPredicate<>(script, resource.getDescription());
        }
    }

    /**
     * Factory to create {@link ScriptedPredicate} from a {@link Resource}.
     * 
     * @param <T> input type
     * @param resource the resource to look at
     * 
     * @return the predicate
     * 
     * @throws ScriptException if the compile fails
     * @throws IOException if the file doesn't exist.
     * @throws ComponentInitializationException if the scripting initialization fails
     */
    @Nonnull public static <T> ScriptedPredicate<T> resourceScript(
            @Nonnull @ParameterName(name="resource") final Resource resource)
                    throws ScriptException, IOException, ComponentInitializationException {
        return resourceScript(DEFAULT_ENGINE, resource);
    }

    /**
     * Factory to create {@link ScriptedPredicate} from inline data.
     *
     * @param <T> input type
     * @param scriptSource the script, as a string
     * @param engineName the language
     * 
     * @return the predicate
     * 
     * @throws ScriptException if the compile fails
     * @throws ComponentInitializationException if the scripting initialization fails
     */
    @Nonnull public static <T> ScriptedPredicate<T> inlineScript(
            @Nonnull @NotEmpty @ParameterName(name="engineName") final String engineName,
            @Nonnull @NotEmpty @ParameterName(name="scriptSource") final String scriptSource)
                    throws ScriptException, ComponentInitializationException {
        final EvaluableScript script = new EvaluableScript();
        script.setEngineName(engineName);
        script.setScript(scriptSource);
        script.initialize();
        return new ScriptedPredicate<>(script, "Inline");
    }

    /**
     * Factory to create {@link ScriptedPredicate} from inline data.
     *
     * @param <T> input type
     * @param scriptSource the script, as a string
     * 
     * @return the predicate
     * 
     * @throws ScriptException if the compile fails
     * @throws ComponentInitializationException if the scripting initialization fails
     */
    @Nonnull public static <T> ScriptedPredicate<T> inlineScript(
            @Nonnull @NotEmpty @ParameterName(name="scriptSource") final String scriptSource)
                    throws ScriptException, ComponentInitializationException {
        final EvaluableScript script = new EvaluableScript();
        script.setEngineName(DEFAULT_ENGINE);
        script.setScript(scriptSource);
        script.initialize();
        return new ScriptedPredicate<>(script, "Inline");
    }

}