/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.security;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;

import com.google.common.net.UrlEscapers;

import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.component.AbstractInitializableComponent;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.net.CookieManager;
import net.shibboleth.shared.net.URISupport;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;

/**
 * Wrapper for managing a cookie encrypted with the {@link DataSealer} component.
 * 
 * <p>The component can be wired up without the necessary components, but then all operations do nothing.
 * This is allowed for the case where deployers disable the {@link DataSealer} component, though very rare.</p>
 *      
 * @since 9.2.0
 */
public class EncryptedCookieManager extends AbstractInitializableComponent {

    /** A negative signal to allow caching opt-out. */
    @Nonnull @NotEmpty public static final String NEGATIVE_VALUE = "__NO";
    
    /** Class logger.*/
    @Nonnull private final Logger log = LoggerFactory.getLogger(EncryptedCookieManager.class);
    
    /** Passwordless cookie name. */
    @Nullable @NotEmpty private String cookieName;
    
    /** Optional cookie manager to use. */
    @Nullable private CookieManager cookieManager;

    /** Optional data sealer to use. */
    @Nullable private DataSealer dataSealer;
    
    /** Flags whether the component is active or should no-op. */
    private boolean active;
    
    /**
     * Set cookie name to use.
     * 
     * @param name cookie name
     */
    public void setCookieName(@Nullable final String name) {
        checkSetterPreconditions();

        cookieName = StringSupport.trimOrNull(name);
    }

    /**
     * Sets {@link CookieManager} to use.
     * 
     * @param manager cookie manager
     */
    public void setCookieManager(@Nullable final CookieManager manager) {
        checkSetterPreconditions();
        
        cookieManager = manager;
    }

    /**
     * Sets {@link DataSealer} to use.
     * 
     * @param sealer data sealer
     */
    public void setDataSealer(@Nullable final DataSealer sealer) {
        checkSetterPreconditions();
        
        dataSealer = sealer;
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        active = cookieName != null && cookieManager != null && dataSealer != null;
    }
    
    /**
     * Tests whether the cookie's value indicates a cached negative response.
     * 
     * @return true iff the input value corresponds to the "opt-out" constant
     */
    public boolean isOptOut() {
        checkComponentActive();

        if (!active) {
            return false;
        }
        
        assert cookieManager != null;
        assert cookieName != null;
        final String value = cookieManager.getCookieValue(cookieName, null);
        
        return NEGATIVE_VALUE.equals(value);
    }
    
    /**
     * Read back existing cookie and return the value embedded in it, if any.
     * 
     * <p>A null is returned in the event of various decoding errors or if the cookie
     * contains the "negative" magic value.</p>
     * 
     * @return value from sealed cookie, or null
     */
    @Nullable @NotEmpty public String readCookie() {
        checkComponentActive();
        
        if (!active) {
            return null;
        }
        
        assert cookieManager != null;
        assert cookieName != null;
        String wrapped = cookieManager.getCookieValue(cookieName, null);
        if (wrapped == null) {
            return null;
        }
        
        if (NEGATIVE_VALUE.equals(wrapped)) {
            return null;
        }
        
        wrapped = URISupport.doURLDecode(wrapped);
        if (wrapped == null) {
            log.error("Error decoding unwrapped cookie value");
            return null;
        }
        
        try {
            assert dataSealer != null;
            return dataSealer.unwrap(wrapped);
        } catch (final DataSealerException e) {
            log.warn("Unable to unwrap sealed cookie", e);
        }
        
        return null;
    }
    
    /**
     * Creates a fresh cookie for a given value (or a placeholder if null to indicate the negative).
     * 
     * @param value value or null
     * 
     * @return true iff the operation succeeded
     */
    public boolean writeCookie(@Nullable final String value) {
        checkComponentActive();
        
        if (!active) {
            return false;
        }
        
        if (value == null || NEGATIVE_VALUE.equals(value)) {
            assert cookieManager != null;
            assert cookieName != null;
            cookieManager.addCookie(cookieName, NEGATIVE_VALUE);
            return true;
        }
        
        try {
            assert dataSealer != null;
            String wrapped = dataSealer.wrap(value);
            wrapped = UrlEscapers.urlFormParameterEscaper().escape(wrapped);
            assert cookieManager != null;
            assert cookieName != null;
            assert wrapped != null;
            cookieManager.addCookie(cookieName, wrapped);
            return true;
        } catch (final DataSealerException e) {
            log.warn("Unable to wrap value for cookie", e);
            return false;
        }
    }

    /**
     * For a non-negative cookie, this recreates the cookie using the current default key to ensure it can
     * continue to be read.
     * 
     * @return true iff the operation succeeded
     */
    public boolean refreshCookie() {
        checkComponentActive();

        if (!active) {
            return false;
        }
        
        final String value = readCookie();
        if (value != null) {
            return writeCookie(value);
        }
        
        return true;
    }
    
    /**
     * Unset the cookie.
     */
    public void clearCookie() {
        checkComponentActive();

        if (!active) {
            return;
        }

        assert cookieManager != null;
        assert cookieName != null;
        cookieManager.unsetCookie(cookieName);
    }
    
    
}