/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.servlet.impl;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;

import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.component.AbstractInitializableComponent;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.servlet.HttpServletRequestValidator;

/**
 * Implementation for a chain of {@link HttpServletRequestValidator}.
 * 
 * @since 9.1.4
 */
public class ChainingHttpServletRequestValidator extends AbstractInitializableComponent
        implements HttpServletRequestValidator {
   
    /** Logger. */
    private Logger log = LoggerFactory.getLogger(ChainingHttpServletRequestValidator.class);
    
    /** Chain validators. */
    @Nonnull private List<HttpServletRequestValidator> validators = CollectionSupport.emptyList();
    
    /**
     * Get the list of registered validators.
     * 
     * @return the list of validators
     */
    @Nonnull @Unmodifiable @NotLive public List<HttpServletRequestValidator> getValidators() {
        return validators;
    }

    /**
     * Set the list of registered validators.
     * 
     * @param newValidators the validators to use
     */
    public void setValidators(@Nullable final List<HttpServletRequestValidator> newValidators) {
        checkSetterPreconditions();
        
        if (newValidators == null) {
            validators = CollectionSupport.emptyList();
        } else {
            validators = newValidators.stream()
                    .filter(Objects::nonNull)
                    .collect(CollectionSupport.nonnullCollector(Collectors.toUnmodifiableList())).get();
        }

    }

    /** {@inheritDoc} */
    @Override
    public void validate(@Nonnull final HttpServletRequest request) throws ServletException {
        for (final HttpServletRequestValidator validator : getValidators()) {
            try {
                validator.validate(request);
            } catch (final ServletException e) {
                log.debug("Request failed validation for validator: {}", validator.getClass().getName());
                throw e;
            }
        }
    }

}
