/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.servlet.impl;

import java.util.Collection;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.component.AbstractInitializableComponent;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.servlet.HttpServletRequestValidator;

/**
 * Component that validates the HTTP request method as one of an allowed set.
 * 
 * @since 9.2.0
 */
public class BasicHttpServletRequestMethodValidator extends AbstractInitializableComponent
        implements HttpServletRequestValidator {
    
    /** Allowed parameters. */
    @Nonnull private Set<String> allowedMethods;
    
    /**
     * Constructor.
     */
    public BasicHttpServletRequestMethodValidator() {
        allowedMethods = CollectionSupport.emptySet();
    }
    
    /**
     * Get the allowed methods. 
     * 
     * @return the allowed methods
     */
    @Nonnull @Unmodifiable @NotLive public Set<String> getAllowedMethods() {
        return allowedMethods;
    }

    /**
     * Set the allowed methods.
     * 
     * @param methods the allowed methods
     */
    public void setAllowedMethods(@Nullable final Collection<String> methods) {
        checkSetterPreconditions();
        allowedMethods = CollectionSupport.copyToSet(StringSupport.normalizeStringCollection(methods));
    }

    /** {@inheritDoc} */
    public void validate(@Nonnull final HttpServletRequest request) throws ServletException {
        Constraint.isNotNull(request, "HttpServletRequest was null");
        
        if (!allowedMethods.contains(request.getMethod())) {
            throw new ServletException("Request method '" + request.getMethod() + "' not in allowed set.");
        }
    }
    
}