/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.servlet.impl;

import java.util.Collection;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.base.Predicates;
import com.google.common.net.MediaType;

import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.component.AbstractInitializableComponent;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.servlet.HttpServletRequestValidator;
import net.shibboleth.shared.servlet.HttpServletSupport;

/**
 * Component that validates the HTTP content type as one of an allowed set.
 * 
 * <p>Optionally allows for no type to be specified.</p>
 * 
 * @since 9.2.0
 */
public class BasicHttpServletRequestContentTypeValidator extends AbstractInitializableComponent
        implements HttpServletRequestValidator {
    
    /** Allowed parameters. */
    @Nonnull private Set<MediaType> allowedContentTypes;
    
    /** Whether an absent content type is allowed. */
    private boolean allowNullContentType;
    
    /**
     * Constructor.
     */
    public BasicHttpServletRequestContentTypeValidator() {
        allowedContentTypes = CollectionSupport.emptySet();
    }
    
    /**
     * Get the allowed types. 
     * 
     * @return the allowed types
     */
    @Nonnull @Unmodifiable @NotLive public Set<String> getAllowedContentTypes() {
        return allowedContentTypes.stream()
                .map(MediaType::toString)
                .collect(CollectionSupport.nonnullCollector(Collectors.toUnmodifiableSet())).get();
    }

    /**
     * Set the allowed types.
     * 
     * @param types the allowed types
     */
    public void setAllowedContentTypes(@Nullable final Collection<String> types) {
        checkSetterPreconditions();
        if (types != null) {
            allowedContentTypes = types.stream()
                    .map(StringSupport::trimOrNull)
                    .filter(Predicates.notNull())
                    .map(MediaType::parse)
                    .collect(CollectionSupport.nonnullCollector(Collectors.toUnmodifiableSet())).get();
        } else {
            allowedContentTypes = CollectionSupport.emptySet();
        }
    }

    /**
     * Gets whether to allow an absent content type.
     * 
     * @return whether to allow an absent content type
     */
    public boolean getAllowNullContentType() {
        return allowNullContentType;
    }
    
    /**
     * Sets whether to allow an absent content type.
     * 
     * <p>Defaults to false.</p>
     * 
     * @param flag flag to set
     */
    public void setAllowNullContentType(final boolean flag) {
        allowNullContentType = flag;
    }
    
    /** {@inheritDoc} */
    public void validate(@Nonnull final HttpServletRequest request) throws ServletException {
        Constraint.isNotNull(request, "HttpServletRequest was null");

        if (!HttpServletSupport.validateContentType(request, allowedContentTypes, allowNullContentType, false)) {
            throw new ServletException("Content type not in allowed set.");
        }
    }
    
}