/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.servlet;

import java.io.IOException;
import java.util.function.Predicate;

import javax.annotation.Nonnull;

import jakarta.servlet.Filter;
import jakarta.servlet.FilterChain;
import jakarta.servlet.FilterConfig;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.logic.PredicateSupport;


/**
 * Base class for HTTP servlet {@link Filter} that determines whether to run dynamically
 * based on a supplied {@link Predicate} instead of based on mapping rules defined in web.xml.
 * 
 * @since 9.0.0
 */
public abstract class AbstractConditionalFilter implements Filter {
    
    /** Whether filter should run or not. */
    @Nonnull private Predicate<ServletRequest> activationCondition;
    
    /** Constructor. */
    public AbstractConditionalFilter() {
        activationCondition = PredicateSupport.alwaysTrue();
    }
    
    /**
     * Get the condition to control activation of this filter.
     * 
     * @return condition
     */
    @Nonnull public Predicate<ServletRequest> getActivationCondition() {
        return activationCondition;
    }
    
    /**
     * Set the condition to control activation of this filter.
     * 
     * @param condition run condition
     */
    public void setActivationCondition(@Nonnull final Predicate<ServletRequest> condition) {
        activationCondition = Constraint.isNotNull(condition, "Run condition cannot be null");
    }
    
    /** {@inheritDoc} */
    public void init(final FilterConfig filterConfig) throws ServletException {
    }

    /** {@inheritDoc} */
    public void destroy() {
    }

    /** {@inheritDoc} */
    public void doFilter(final ServletRequest request, final ServletResponse response, final FilterChain chain)
            throws IOException, ServletException {
        
        assert request != null;
        assert response != null;
        assert chain != null;
        
        if (activationCondition.test(request)) {
            runFilter(request, response, chain);
            return;
        }

        chain.doFilter(request, response);
    }
    
    /**
     * Subclasses should override this method to be called when the filter is directed to activate.
     * 
     * @param request servlet request
     * @param response servlet response
     * @param chain filter chain
     * 
     * @throws ServletException on error 
     * @throws IOException on error
     */
    protected abstract void runFilter(@Nonnull final ServletRequest request, @Nonnull final ServletResponse response,
            @Nonnull final FilterChain chain) throws IOException, ServletException;
    
}