/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.net;

import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;

import jakarta.servlet.http.Cookie;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import net.shibboleth.shared.annotation.constraint.NonNegative;
import net.shibboleth.shared.annotation.constraint.NonnullAfterInit;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.component.AbstractInitializableComponent;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.logic.PredicateSupport;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.NonnullSupplier;
import net.shibboleth.shared.primitive.StringSupport;

/**
 * A helper class for managing one or more cookies on behalf of a component.
 * 
 * <p>This bean centralizes settings related to cookie creation and access,
 * and is parameterized by name so that multiple cookies may be managed with
 * common properties.</p>
 * 
 * <p>Some of the features depend on Servlet API >= 6.0 and will be conditionally
 * bypassed otherwise.</p>
 */
public class CookieManager extends AbstractInitializableComponent {
    
    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(CookieManager.class);

    /** Path of cookie. */
    @Nullable private String cookiePath;

    /** Domain of cookie. */
    @Nullable private String cookieDomain;
    
    /** Supplier for the servlet request to read from. */
    @NonnullAfterInit private NonnullSupplier<HttpServletRequest> httpRequestSupplier;

    /** Supplier for the servlet response to write to. */
    @NonnullAfterInit private NonnullSupplier<HttpServletResponse> httpResponseSupplier;
    
    /** Is cookie secure? */
    private boolean secure;

    /** Is cookie marked HttpOnly? */
    private boolean httpOnly;
    
    /** Maximum age in seconds, or -1 for session. */
    private int maxAge;
    
    /** Whether to guard {@link Cookie#setAttribute(String, String)} calls. */
    private boolean guardSetAttribute;
    
    /** The allowed same-site cookie attribute values.*/
    public enum SameSiteValue{       
        
        /**
         * Send the cookie for 'same-site' requests only.
         */
        Strict("Strict"),
        /**
         * Send the cookie for 'same-site' requests along with 'cross-site' top 
         * level navigations using safe HTTP methods (GET, HEAD, OPTIONS, and TRACE).
         */
        Lax("Lax"),        
        /**
         * Send the cookie for 'same-site' and 'cross-site' requests.
         */
        None("None"),
        /**
         * Specify nothing.
         */
        Null("Null");

        /** The same-site attribute value.*/
        @Nonnull @NotEmpty private String value;
        
        /**
         * Constructor.
         *
         * @param attrValue the same-site attribute value.
         */
        private SameSiteValue(@Nonnull @NotEmpty final String attrValue) {
            value = Constraint.isNotEmpty(attrValue, "the same-site attribute value can not be empty");
         }

        /**
         * Get the same-site attribute value.
         * 
         * @return Returns the value.
         */
        @Nonnull public String getValue() {
            return value;
        }
    }
    
    /** SameSite attribute. */
    @Nonnull private SameSiteValue defaultSameSite;
    
    /** Map of cookie name to same-site attribute value.*/
    @Nonnull private Map<String,SameSiteValue> sameSiteCookies;
    
    /** Condition controlling application of SameSite. */
    @Nonnull private Predicate<HttpServletRequest> sameSiteCondition;
    
    /** Additional cookie attributes. */
    @Nonnull private Map<String,String> cookieAttributes;
    
    /** Limit on numbeer of cookies when purging. */
    @NonNegative private int cookieLimit;
    
    /** Constructor. */
    public CookieManager() {
        httpOnly = true;
        secure = true;
        maxAge = -1;
        defaultSameSite = SameSiteValue.Null;
        sameSiteCookies = CollectionSupport.emptyMap();
        sameSiteCondition = PredicateSupport.alwaysTrue();
        cookieAttributes = CollectionSupport.emptyMap();
        cookieLimit = 0;
        guardSetAttribute = true;
    }

    /**
     * Get the cookie path to use.
     * 
     * @return cookie path
     * 
     * @since 9.1.0
     */
    @Nullable public String getCookiePath() {
        return cookiePath;
    }
    
    /**
     * Set the cookie path to use.
     * 
     * <p>Defaults to the servlet context path.</p>
     * 
     * @param path cookie path to use, or null for the default
     */
    public void setCookiePath(@Nullable final String path) {
        checkSetterPreconditions();
        
        cookiePath = StringSupport.trimOrNull(path);
    }

    /**
     * Get the cookie domain to use.
     * 
     * @return cookie domain
     * 
     * @since 9.1.0
     */
    @Nullable public String getCookieDomain() {
        return cookieDomain;
    }
    
    /**
     * Set the cookie domain to use.
     * 
     * @param domain the cookie domain to use, or null for the default
     */
    public void setCookieDomain(@Nullable final String domain) {
        checkSetterPreconditions();
        
        cookieDomain = StringSupport.trimOrNull(domain);
    }

    /**
     * Set the Supplier for the servlet request to read from.
     *
     * @param requestSupplier servlet request supplier
     */
    public void setHttpServletRequestSupplier(@Nonnull final NonnullSupplier<HttpServletRequest> requestSupplier) {
        checkSetterPreconditions();
        httpRequestSupplier = Constraint.isNotNull(requestSupplier, "HttpServletRequest cannot be null");
    }

    /**
     * Get the current HTTP request if available.
     *
     * @return current HTTP request
     */
    @NonnullAfterInit protected HttpServletRequest getHttpServletRequest() {
        if (httpRequestSupplier == null) {
            return null;
        }
        return httpRequestSupplier.get();
    }

    /**
     * Set the supplier for the servlet response to write to.
     *
     * @param responseSupplier servlet response
     */
    public void setHttpServletResponseSupplier(@Nonnull final NonnullSupplier<HttpServletResponse> responseSupplier) {
        checkSetterPreconditions();
        httpResponseSupplier = Constraint.isNotNull(responseSupplier, "HttpServletResponse cannot be null");
    }

    /**
     * Get the current HTTP response if available.
     *
     * @return current HTTP response or null
     */
    @NonnullAfterInit protected HttpServletResponse getHttpServletResponse() {
        if (httpResponseSupplier == null) {
            return null;
        }
        return httpResponseSupplier.get();
    }

    /**
     * Get the TLS-only flag.
     * 
     * @return TLS-only flag
     * 
     * @since 9.2.0
     */
    public boolean isSecure() {
        return secure;
    }

    /**
     * Set the TLS-only flag.
     * 
     * @param flag flag to set
     */
    public void setSecure(final boolean flag) {
        checkSetterPreconditions();
        
        secure = flag;
    }
    
    /**
     * Get the HttpOnly flag.
     * 
     * @return HttpOnly flag
     * 
     * @since 9.2.0
     */
    public boolean isHttpOnly() {
        return httpOnly;
    }

    /**
     * Set the HttpOnly flag.
     * 
     * @param flag flag to set
     */
    public void setHttpOnly(final boolean flag) {
        checkSetterPreconditions();

        httpOnly = flag;
    }
    
    /**
     * Get maximum age of cookies in seconds, or -1 for a session cookie.
     * 
     * <p>Note that this "convention" for -1 is NOT consistent with the RFC and is a hold-over from
     * Java's native Cookie class, probably because the Integer type didn't exist, so there was no way 
     * o easily handle a null Max-Age value in the API.</p>
     * 
     * @return max age in seconds, or -1 for a session cookie
     * 
     * @since 9.1.0
     */
    public int getMaxAge() {
        return maxAge;
    }
    
    /**
     * Maximum age in seconds, or -1 for a session cookie.
     * 
     * <p>Note that this "convention" for -1 is NOT consistent with the RFC and is a hold-over from
     * Java's native Cookie class, probably because the Integer type didn't exist, so there was no way 
     * o easily handle a null Max-Age value in the API.</p>
     * 
     * @param age max age to set
     */
    public void setMaxAge(final int age) {
        checkSetterPreconditions();

        maxAge = age;
    }
    
    /**
     * Maximum age expressed as a {@link Duration}.
     * 
     * @param age max age of cookie
     * 
     * @since 9.0.0
     */
    public void setMaxAgeDuration(@Nonnull final Duration age) {
        checkSetterPreconditions();
        
        maxAge = (int) Constraint.isNotNull(age, "Age cannot be null").getSeconds();
        if (maxAge < 0) {
            maxAge = -1;
        }
    }
    
    /**
     * Sets whether to guard calls to {@link Cookie#setAttribute(String, String)} with a check for
     * the container's servlet API version.
     * 
     * <p>Defaults to true.</p>
     * 
     * @param flag
     */
    public void setGuardSetAttribute(final boolean flag) {
        checkSetterPreconditions();
        
        guardSetAttribute = flag;
    }

    /**
     * Gets the SameSite attribute.
     * 
     * @return the SameSite attribute
     * 
     * @since 9.2.0
     */
    @Nonnull public SameSiteValue getSameSite() {
        return defaultSameSite;
    }

    /**
     * Sets the SameSite attribute.
     * 
     * <p>Defaults to non-existent (which is not the same as "None").</p>
     * 
     * @param value value to set
     * 
     * @since 9.2.0
     */
    public void setSameSite(@Nonnull final SameSiteValue value) {
        checkSetterPreconditions();
        
        defaultSameSite = Constraint.isNotNull(value, "SameSite Value cannot be null");
    }
    
    /**
     * Set the names of cookies to add the same-site attribute to. 
     * 
     * <p>The argument map is flattened to remove the nested collection. The argument map allows duplicate 
     * cookie names to appear in order to detect configuration errors which would otherwise not be found during 
     * argument injection e.g. trying to set a session identifier cookie as both SameSite=Strict and SameSite=None. 
     * Instead, duplicates are detected here, throwing a terminating {@link IllegalArgumentException} if found.</p>
     * 
     * @param map the map of same-site attribute values to cookie names
     * 
     * @since 9.2.0
     */
    public void setSameSiteCookies(@Nullable final Map<SameSiteValue,List<String>> map) {
        if (map != null) {
            sameSiteCookies = new HashMap<>(4);
            for (final Map.Entry<SameSiteValue,List<String>> entry : map.entrySet()) {
                
                for (final String cookieName : entry.getValue()) {
                   if (sameSiteCookies.get(cookieName) != null) {
                       log.error("Duplicate cookie name '{}' found in SameSite cookie map, "
                               + "please check configuration.",cookieName);
                       throw new IllegalArgumentException("Duplicate cookie name found in SameSite cookie map");
                   }  
                   final String trimmedName = StringSupport.trimOrNull(cookieName);
                    if (trimmedName != null) {
                        sameSiteCookies.put(cookieName, entry.getKey());
                    }
                }                
            }
        } else {
            sameSiteCookies = CollectionSupport.emptyMap();
        }
        
    }
    
    /**
     * Gets the condition controlling application of SameSite.
     * 
     * @return condition
     * 
     * @since 9.2.0
     */
    @Nonnull public Predicate<HttpServletRequest> getSameSiteCondition() {
        return sameSiteCondition;
    }
    
    /**
     * Sets the condition controlling application of SameSite.
     * 
     * <p>Defaults to true.</p>
     * 
     * @param condition condition to set
     * 
     * @since 9.2.0
     */
    public void setSameSiteCondition(@Nonnull final Predicate<HttpServletRequest> condition) {
        checkSetterPreconditions();
        
        sameSiteCondition = Constraint.isNotNull(condition, "SameSite condition cannot be null");
    }
    
    /**
     * Gets additional attributes to apply to the cookie.
     * 
     * @return additional attributes
     * 
     * @since 9.2.0
     */
    @Nonnull @NotLive @Unmodifiable public Map<String,String> getCookieAttributes() {
        return cookieAttributes;
    }
    
    /**
     * Sets additional attributes to apply to the cookie.
     * 
     * @param attributes attributes to apply
     * 
     * @since 9.2.0
     */
    public void setCookieAttributes(@Nullable final Map<String,String> attributes) {
        checkSetterPreconditions();
        
        if (attributes != null) {
            cookieAttributes = CollectionSupport.copyToMap(attributes);
        } else {
            cookieAttributes = CollectionSupport.emptyMap();
        }
    }
    
    /**
     * Gets the limit on cookies of a given set when purging.
     * 
     * @return the limit or 0 for unlimited
     * 
     * @since 9.2.0
     */
    @NonNegative public int getCookieLimit() {
        return cookieLimit;
    }

    /**
     * Sets the limit on cookies of a given set when purging.
     * 
     * <p>Defaults to 0, no limit.</p>
     * 
     * @param limit limit to set or 0 for unlimited
     * 
     * @since 9.2.0
     */
    public void setCookieLimit(@NonNegative final int limit) {
        checkSetterPreconditions();
        
        cookieLimit = Constraint.isGreaterThanOrEqual(0, limit, "");
    }

    /** {@inheritDoc} */
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (httpRequestSupplier == null || httpResponseSupplier == null) {
            throw new ComponentInitializationException("Servlet request and response suppliers must be set");
        }
    }

    /**
     * Add a cookie with the specified attributes.
     * 
     * @param name  name of cookie
     * @param value value of cookie
     */
    public void addCookie(@Nonnull @NotEmpty final String name, @Nonnull @NotEmpty final String value) {
        addCookie(name, value, getCookiePath(), getMaxAge());
    }
    
    /**
     * Add a cookie with the specified attributes.
     * 
     * @param name  name of cookie
     * @param value value of cookie
     * @param overrideMaxAge max-age value to use
     * 
     * @since 9.1.0
     */
    public void addCookie(@Nonnull @NotEmpty final String name, @Nonnull @NotEmpty final String value,
            final int overrideMaxAge) {
        addCookie(name, value, getCookiePath(), overrideMaxAge);
    }

    /**
     * Add a cookie with the specified attributes.
     * 
     * @param name  name of cookie
     * @param value value of cookie
     * @param overridePath path value tp use
     * @param overrideMaxAge max-age value to use
     * 
     * @since 9.2.0
     */
    public void addCookie(@Nonnull @NotEmpty final String name, @Nonnull @NotEmpty final String value,
            @Nullable @NotEmpty final String overridePath, final int overrideMaxAge) {
        checkComponentActive();
        
        final Cookie cookie = new Cookie(name, value);
        cookie.setPath(overridePath  != null ? overridePath : contextPathToCookiePath());
        if (getCookieDomain() != null) {
            cookie.setDomain(getCookieDomain());
        }
        cookie.setSecure(isSecure());
        cookie.setHttpOnly(isHttpOnly());
        cookie.setMaxAge(overrideMaxAge);
        if (!guardSetAttribute || getHttpServletRequest().getServletContext().getMajorVersion() >= 6) {
            attachSameSite(cookie);
            cookieAttributes.forEach((n,v) -> {
                cookie.setAttribute(n, v);
                });
        }
        
        getHttpServletResponse().addCookie(cookie);
    }

    /**
     * Unsets a cookie with the specified name and the default path.
     * 
     * @param name  name of cookie
     */
    public void unsetCookie(@Nonnull @NotEmpty final String name) {
        unsetCookie(name, getCookiePath());
    }

    /**
     * Unsets a cookie with the specified name and path.
     * 
     * @param name  name of cookie
     * @param overridePath cookie path
     */
    public void unsetCookie(@Nonnull @NotEmpty final String name, @Nullable @NotEmpty final String overridePath) {
        checkComponentActive();
        
        final Cookie cookie = new Cookie(name, null);
        cookie.setPath(overridePath != null ? overridePath : contextPathToCookiePath());
        if (cookieDomain != null) {
            cookie.setDomain(getCookieDomain());
        }
        cookie.setSecure(isSecure());
        cookie.setHttpOnly(isHttpOnly());
        cookie.setMaxAge(0);
        if (!guardSetAttribute || getHttpServletRequest().getServletContext().getMajorVersion() >= 6) {
            attachSameSite(cookie);
            cookieAttributes.forEach((n,v) -> {
                cookie.setAttribute(n, v);
                });
        }
        
        getHttpServletResponse().addCookie(cookie);
    }
    
    /**
     * Check whether a cookie has a certain value.
     * 
     * @param name name of cookie
     * @param expectedValue expected value of cookie
     * 
     * @return true iff the cookie exists and has the expected value
     */
    public boolean cookieHasValue(@Nonnull @NotEmpty final String name, @Nonnull @NotEmpty final String expectedValue) {
        
        final String realValue =  getCookieValue(name, null);
        if (realValue == null) {
            return false;
        }
        
        return realValue.equals(expectedValue);
    }
    
    /**
     * Return the first matching cookie's value.
     * 
     * @param name cookie name
     * @param defValue default value to return if the cookie isn't found
     * 
     * @return cookie value
     */
    @Nullable public String getCookieValue(@Nonnull @NotEmpty final String name, @Nullable final String defValue) {
        checkComponentActive();
        
        final Cookie[] cookies = getHttpServletRequest().getCookies();
        if (cookies != null) {
            for (final Cookie cookie : cookies) {
                if (cookie.getName().equals(name)) {
                    return cookie.getValue();
                }
            }
        }
        
        return defValue;
    }

    /**
     * Unset cookies matching a given prefix in excess of the configured amount. 
     * 
     * @param prefix cookie name prefix to match on
     * 
     * @since 9.2.0
     */
    public void purgeStaleCookies(@Nonnull @NotEmpty final String prefix) {
        purgeStaleCookies(prefix, getCookiePath());
    }
    
    /**
     * Unset cookies matching a given prefix in excess of the configured amount. 
     * 
     * @param prefix cookie name prefix to match on
     * @param overridePath cookie path
     * 
     * @since 9.2.0
     */
    public void purgeStaleCookies(@Nonnull @NotEmpty final String prefix,
            @Nullable @NotEmpty final String overridePath) {
        
        if (getCookieLimit() == 0) {
            return;
        }
        
        final Cookie[] cookies = getHttpServletRequest().getCookies();
        if (cookies == null || cookies.length == 0 || cookies.length <= getCookieLimit()) {
            return;
        }
        
        // Build a list of matching names with the specified prefix we can sort.
        final TreeSet<String> sortedNames = new TreeSet<>();
        for (final Cookie c : cookies) {
            if (c.getName().startsWith(prefix)) {
                sortedNames.add(c.getName());
            }
        }

        int maxCookies = getCookieLimit();
        int purgedCookies = 0;
        
        for (final String nameToPurge : sortedNames.descendingSet()) {
            if (maxCookies > 0) {
                // Keep it but count against limit.
                --maxCookies;
            } else {
                // We're over the limit, so everything here and older gets cleaned up.
                unsetCookie(nameToPurge, overridePath);
                ++purgedCookies;
            }
        }
        
        if (purgedCookies > 0) {
            log.debug("Purged {} stale cookie(s) with prefix '{}'", purgedCookies, prefix);
        }
    }
    
    /**
     * Implementation of SameSite attachment logic when available.
     * 
     * @param cookie cookie to attach attribute to
     */
    private void attachSameSite(@Nonnull final Cookie cookie) {
        
        if (!sameSiteCondition.test(getHttpServletRequest())) {
            return;
        }
        
        final SameSiteValue sameSiteValue = sameSiteCookies.get(cookie.getName());
        if (sameSiteValue != null) {
            if (sameSiteValue != SameSiteValue.Null) {
                cookie.setAttribute("SameSite", sameSiteValue.getValue());
            }
        } else if (defaultSameSite != SameSiteValue.Null) {
            cookie.setAttribute("SameSite", defaultSameSite.getValue());
        }
    }
    
    /**
     * Turn the servlet context path into an appropriate cookie path.
     * 
     * @return  the cookie path
     */
    @Nonnull @NotEmpty private String contextPathToCookiePath() {
        final  HttpServletRequest httpRequest = getHttpServletRequest();
        return "".equals(httpRequest.getContextPath()) ? "/" : httpRequest.getContextPath();
    }
    
}
