/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.servlet.impl;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import net.shibboleth.shared.logic.Constraint;

/**
 * Class which holds and makes available the current HTTP servlet request and response via ThreadLocal storage.
 * 
 * <p>
 * See also net.shibboleth.shared.spring.servlet.impl.RequestResponseContextFilter, a Java Servlet
 * {@link jakarta.servlet.Filter}-based way to populate and clean up this context in a servlet container.
 * </p>
 */
public final class HttpServletRequestResponseContext {

    /** ThreadLocal storage for request. */
    @Nonnull private static final ThreadLocal<HttpServletRequest> CURRENT_REQUEST = new ThreadLocal<>();

    /** ThreadLocal storage for response. */
    @Nonnull private static final ThreadLocal<HttpServletResponse> CURRENT_RESPONSE = new ThreadLocal<>();

    /** Constructor. */
    private HttpServletRequestResponseContext() {
    };

    /**
     * Load the thread-local storage with the current request and response.
     * 
     * @param request the current {@link HttpServletRequest}
     * @param response the current {@link HttpServletResponse}
     */
    public static void loadCurrent(@Nonnull final HttpServletRequest request,
            @Nonnull final HttpServletResponse response) {
        Constraint.isNotNull(request, "HttpServletRequest may not be null");
        Constraint.isNotNull(response, "HttpServletResponse may not be null");

        CURRENT_REQUEST.set(request);
        CURRENT_RESPONSE.set(response);
    }

    /**
     * Clear the current thread-local context instances.
     */
    public static void clearCurrent() {
        CURRENT_REQUEST.remove();
        CURRENT_RESPONSE.remove();
    }

    /**
     * Get the current {@link HttpServletRequest} being serviced by the current thread.
     * 
     * @return the current request
     */
    @Nullable public static HttpServletRequest getRequest() {
        return CURRENT_REQUEST.get();
    }

    /**
     * Get the current {@link HttpServletResponse} being serviced by the current thread.
     * 
     * @return the current response
     */
    @Nullable public static HttpServletResponse getResponse() {
        return CURRENT_RESPONSE.get();
    }

}