/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.net;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.shared.annotation.constraint.Live;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.NullableElements;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.collection.LazyList;
import net.shibboleth.shared.collection.Pair;
import net.shibboleth.shared.primitive.StringSupport;

/** Helper methods for building {@link URI}s and parsing some HTTP URL information. */
public final class URISupport {

    /** Constructor. */
    private URISupport() {
    }

    /**
     * Sets the fragment of a URI.
     * 
     * @param prototype prototype URI that provides information other than the fragment
     * @param fragment fragment for the new URI
     * 
     * @return new URI built from the prototype URI and the given fragment
     */
    @Nonnull public static URI setFragment(@Nonnull final URI prototype, @Nullable final String fragment) {
        try {
            return new URI(prototype.getScheme(), prototype.getUserInfo(), prototype.getHost(), prototype.getPort(),
                    prototype.getPath(), prototype.getQuery(), trimOrNullFragment(fragment));
        } catch (final URISyntaxException e) {
            throw new IllegalArgumentException("Illegal fragment text", e);
        }
    }

    /**
     * Sets the host of a URI.
     * 
     * @param prototype prototype URI that provides information other than the host
     * @param host host for the new URI
     * 
     * @return new URI built from the prototype URI and the given host
     */
    @Nonnull public static URI setHost(@Nonnull final URI prototype, @Nullable final String host) {
        try {
            return new URI(prototype.getScheme(), prototype.getUserInfo(), StringSupport.trimOrNull(host),
                    prototype.getPort(), prototype.getPath(), prototype.getQuery(), prototype.getFragment());
        } catch (final URISyntaxException e) {
            throw new IllegalArgumentException("Illegal host", e);
        }
    }

    /**
     * Sets the path of a URI.
     * 
     * @param prototype prototype URI that provides information other than the path
     * @param path path for the new URI
     * 
     * @return new URI built from the prototype URI and the given path
     */
    @Nonnull public static URI setPath(@Nonnull final URI prototype, @Nullable final String path) {
        try {
            return new URI(prototype.getScheme(), prototype.getUserInfo(), prototype.getHost(), prototype.getPort(),
                    trimOrNullPath(path), prototype.getQuery(), prototype.getFragment());
        } catch (final URISyntaxException e) {
            throw new IllegalArgumentException("Illegal path", e);
        }
    }

    /**
     * Sets the port of a URI.
     * 
     * @param prototype prototype URI that provides information other than the port
     * @param port port for the new URI
     * 
     * @return new URI built from the prototype URI and the given port
     */
    @Nonnull public static URI setPort(@Nonnull final URI prototype, final int port) {
        try {
            return new URI(prototype.getScheme(), prototype.getUserInfo(), prototype.getHost(), port,
                    prototype.getPath(), prototype.getQuery(), prototype.getFragment());
        } catch (final URISyntaxException e) {
            throw new IllegalArgumentException("Illegal port", e);
        }
    }

    /**
     * Sets the query of a URI.
     * 
     * <p>
     * <b>WARNING:</b> If the supplied query parameter names and/or values contain '%' characters 
     * (for example because they are already Base64-encoded), then the approach of using {@link URI} 
     * instances to work with the URI/URL may not be appropriate.  Per its documentation, the 
     * {@link URI} constructors always encode '%' characters, which can lead to cases of double-encoding.
     * For an alternative way of manipulating URL's see {@link URLBuilder}.
     * </p>
     * 
     * @param prototype prototype URI that provides information other than the query
     * @param query query for the new URI
     * 
     * @return new URI built from the prototype URI and the given query
     */
    @Nonnull public static URI setQuery(@Nonnull final URI prototype, @Nullable final String query) {
        try {
            return new URI(prototype.getScheme(), prototype.getUserInfo(), prototype.getHost(), prototype.getPort(),
                    prototype.getPath(), trimOrNullQuery(query), prototype.getFragment());
        } catch (final URISyntaxException e) {
            throw new IllegalArgumentException("Illegal query", e);
        }
    }

    /**
     * Sets the query of a URI.
     * 
     * <p>
     * <b>WARNING:</b> If the supplied query parameter names and/or values contain '%' characters 
     * (for example because they are already Base64-encoded), then the approach of using {@link URI} 
     * instances to work with the URI/URL may not be appropriate.  Per its documentation, the 
     * {@link URI} constructors always encode '%' characters, which can lead to cases of double-encoding.
     * For an alternative way of manipulating URL's see {@link URLBuilder}.
     * </p>
     * 
     * <p>Note that while the input may not contain null elements, its actual elements may themselves
     * contain null values (the second half of the pair).</p>
     * 
     * @param prototype prototype URI that provides information other than the query
     * @param parameters query parameters for the new URI
     * 
     * @return new URI built from the prototype URI and the given query
     */
    @Nonnull public static URI setQuery(@Nonnull final URI prototype,
            @Nullable final List<Pair<String, String>> parameters) {
        try {
            return new URI(prototype.getScheme(), prototype.getUserInfo(), prototype.getHost(), prototype.getPort(),
                    prototype.getPath(), buildQuery(parameters), prototype.getFragment());
        } catch (final URISyntaxException e) {
            throw new IllegalArgumentException("Illegal query", e);
        }
    }

    /**
     * Sets the scheme of a URI.
     * 
     * @param prototype prototype URI that provides information other than the scheme
     * @param scheme scheme for the new URI
     * 
     * @return new URI built from the prototype URI and the given scheme
     */
    @Nonnull public static URI setScheme(@Nonnull final URI prototype, @Nullable final String scheme) {
        try {
            return new URI(StringSupport.trimOrNull(scheme), prototype.getUserInfo(), prototype.getHost(),
                    prototype.getPort(), prototype.getPath(), prototype.getQuery(), prototype.getFragment());
        } catch (final URISyntaxException e) {
            throw new IllegalArgumentException("Illegal scheme", e);
        }
    }
    
    /**
     * Create a file: URI from an absolute path, dealing with the Windows, non leading "/" issue.
     *
     * <p>Windows absolute paths have a habit of starting with a "DosDeviceName" (such as <code>C:\absolute\path</code>
     * if we blindly convert that to a file URI by prepending "file://", then we end up with a URI which has "C:" as 
     * the network segment.  So if we need to have an absolute file path based URI (JAAS is the example) we call this
     * method which hides the hideous implementation.</p>
     *
     * @param path the absolute file path to convert
     * @return a suitable URI
     * @throws URISyntaxException if the URI contructor fails
     */
    @Nonnull public static URI fileURIFromAbsolutePath(@Nonnull final String path) throws URISyntaxException {
        final StringBuilder uriPath = new StringBuilder(path.length()+8);
        
        uriPath.append("file://");
        if (!path.startsWith("/")) {
            // it's windows
            uriPath.append('/');
        }
        uriPath.append(path);
        return new URI(uriPath.toString());
    }

    /**
     * Builds an RFC-3968 encoded URL query component from a collection of parameters.
     * 
     * <p>Note that while the input may not contain null elements, its actual elements may themselves
     * contain null values (the second half of the pair).</p>
     * 
     * @param parameters collection of parameters from which to build the URL query component
     * 
     * @return RFC-3968 encoded URL query or null if the parameter collection was null or empty
     */
    @Nullable public static String buildQuery(@Nullable final List<Pair<String, String>> parameters) {
        if (parameters == null || parameters.size() == 0) {
            return null;
        }

        final StringBuilder builder = new StringBuilder();
        boolean firstParam = true;
        for (final Pair<String, String> parameter : parameters) {
            if (firstParam) {
                firstParam = false;
            } else {
                builder.append("&");
            }

            builder.append(doURLEncode(parameter.getFirst()));
            builder.append("=");
            if (parameter.getSecond() != null) {
                builder.append(doURLEncode(parameter.getSecond()));
            }
        }

        return builder.toString();
    }
    
    /**
     * Builds a map from a collection of parameters.
     * 
     * <p>
     * Note that this method only properly supports parameters which have a single value.
     * If support for parameters with multiple values is required, instead use {@link #buildQueryMultiMap(List)}.
     * </p>
     * 
     * @param parameters collection of parameters from which to build the corresponding, may be null or empty
     * 
     * @return a non-null map of query parameter name-&gt; value. Keys will be non-null. Values may be null.
     */
    @Nonnull @NullableElements @Unmodifiable @NotLive public static Map<String,String> buildQueryMap(
            @Nullable @NullableElements final List<Pair<String, String>> parameters) {
        if (parameters == null || parameters.size() == 0) {
            return CollectionSupport.emptyMap();
        }
        
        final HashMap<String,String> map = new HashMap<>();
        for (final Pair<String,String> param : parameters) {
            if (param.getFirst() != null) {
                map.put(param.getFirst(), param.getSecond());
            }
        }
        
        // Allow for null values.
        return CollectionSupport.copyToMap(map);
    }

    /**
     * Builds a map from a collection of parameters.
     * 
     * @param parameters collection of parameters from which to build the corresponding, may be null or empty
     * 
     * @return a non-null map of query parameter name-&gt; value. Keys will be non-null. List values will be non-null,
     *          but may contain null elements
     */
    @Nonnull @NullableElements @Unmodifiable @NotLive public static Map<String,List<String>> buildQueryMultiMap(
            @Nullable @NullableElements final List<Pair<String, String>> parameters) {
        if (parameters == null || parameters.size() == 0) {
            return CollectionSupport.emptyMap();
        }
        
        final HashMap<String,List<String>> map = new HashMap<>();
        for (final Pair<String,String> param : parameters) {
            if (param.getFirst() != null) {
                List<String> currentValue = map.get(param.getFirst());
                if (currentValue == null) {
                    currentValue = new ArrayList<>();
                    map.put(param.getFirst(), currentValue);
                }
                currentValue.add(param.getSecond());
            }
        }
        
        // Allow for null values.
        return CollectionSupport.copyToMap(map);
    }

    /**
     * Get the first raw (i.e.RFC-3968 encoded) query string component with the specified parameter name. This method
     * assumes the common query string format of one or more 'paramName=paramValue' pairs separated by '&amp;'.
     * 
     * The component will be returned as a string in the form 'paramName=paramValue' (minus the quotes).
     * 
     * @param queryString the URL encoded HTTP URL query string
     * @param paramName the URL decoded name of the parameter to find
     * @return the found component, or null if query string or param name is null/empty or the parameter is not found
     */
    @Nullable public static String getRawQueryStringParameter(@Nullable final String queryString,
            @Nullable final String paramName) {
        final String trimmedQuery = trimOrNullQuery(queryString);
        final String trimmedName = StringSupport.trimOrNull(paramName);
        if (trimmedQuery == null || trimmedName == null) {
            return null;
        }

        final String encodedName = doURLEncode(trimmedName);
        
        final String[] candidates = trimmedQuery.split("&");
        for (final String candidate : candidates) {
            if (candidate.startsWith(encodedName+"=") || candidate.equals(encodedName)) {
                return candidate;
            }
        }
        
        return null;
    }
    
    /**
     * Get all raw (i.e.RFC-3968 encoded) query string components with the specified parameter name. This method
     * assumes the common query string format of one or more 'paramName=paramValue' pairs separated by '&amp;'.
     * Parameters without values will be represented in the returned map as a key associated with
     * the value <code>null</code>.
     * 
     * The parameter name match will be performed using the URL decoded forms of both the specified
     * <code>paramName</code> and the candidate query string value.
     * 
     * @param queryString the URL encoded HTTP URL query string
     * @param paramName the URL decoded name of the parameter to find
     * @return the found component, or null if query string or param name is null/empty or the parameter is not found
     */
    @Nonnull @Live public static List<Pair<String, String>> getRawQueryStringParameters(
            @Nullable final String queryString,
            @Nullable final String paramName) {
        
        final String paramNameTrimmed = StringSupport.trimOrNull(paramName);
        if (paramNameTrimmed == null) {
            return CollectionSupport.emptyList();
        }

        return parseQueryString(queryString, false, false).stream()
                .filter(p -> Objects.equals(doURLDecode(paramNameTrimmed), doURLDecode(p.getFirst())))
                .collect(CollectionSupport.nonnullCollector(Collectors.toList())).get();
    }

    /**
     * Parses a RFC-3968 encoded query string in to a set of name/value pairs. This method assumes the common query
     * string format of one or more 'paramName=paramValue' pairs separate by '&amp;'. Both parameter names and values
     * will be URL decoded. Parameters without values will be represented in the returned map as a key associated with
     * the value <code>null</code>.
     * 
     * @param queryString URL encoded query string
     * 
     * @return the parameters from the query string, never null
     */
    @Nonnull @Live public static List<Pair<String, String>> parseQueryString(@Nullable final String queryString) {
        return parseQueryString(queryString, true, true);
    }

    /**
     * Parses a RFC-3968 encoded query string in to a set of name/value pairs. This method assumes the common query
     * string format of one or more 'paramName=paramValue' pairs separate by '&amp;'. URL decoding of parameter
     * names and values is determined by the relevant arguments. Parameters without values will be represented in the
     * returned map as a key associated with the value <code>null</code>.
     * 
     * @param queryString URL encoded query string
     * @param decodeName whether the parameter names should be URL decoded in the returned list
     * @param decodeValue whether the parameter values should be URL decoded in the returned list
     * @param charset character encoding
     * 
     * @return the parameters from the query string, never null
     * 
     * @since 9.2.0
     */
    @Nonnull @Live public static List<Pair<String, String>> parseQueryString(@Nullable final String queryString,
            final boolean decodeName, final boolean decodeValue, @Nonnull final Charset charset) {

        final String trimmedQuery = trimOrNullQuery(queryString);
        if (trimmedQuery == null) {
            return new LazyList<>();
        }

        final ArrayList<Pair<String, String>> queryParams = new ArrayList<>();
        final String[] paramPairs = trimmedQuery.split("&");
        String[] param;
        for (final String paramPair : paramPairs) {
            param = paramPair.split("=");
            if (param.length == 1) {
                queryParams.add(new Pair<>(decodeName ? doURLDecode(param[0], charset) : param[0], (String) null));
            } else {
                queryParams.add(new Pair<>(decodeName ? doURLDecode(param[0]) : param[0],
                        decodeValue ? doURLDecode(param[1], charset) : param[1]));
            }
        }

        return queryParams;
    }
    
    /**
     * Same as {@link #parseQueryString(String, boolean, boolean, Charset)} with the final parameter set
     * to {@link StandardCharsets#UTF_8}.
     * 
     * @param queryString URL encoded query string
     * @param decodeName whether the parameter names should be URL decoded in the returned list
     * @param decodeValue whether the parameter values should be URL decoded in the returned list
     * 
     * @return the parameters from the query string, never null
     */
    @Nonnull @Live public static List<Pair<String, String>> parseQueryString(@Nullable final String queryString,
            final boolean decodeName, final boolean decodeValue) {

        return parseQueryString(queryString, decodeName, decodeValue, StandardCharsets.UTF_8);
    }

    /**
     * Trims an RFC-3968 encoded URL path component. If the given path is null or empty then null is returned. If the
     * given path ends with '?' then it is removed. If the given path ends with '#' then it is removed.
     * 
     * @param path path to trim
     * 
     * @return the trimmed path or null
     */
    @Nullable public static String trimOrNullPath(@Nullable final String path) {
        String trimmedPath = StringSupport.trimOrNull(path);
        if (trimmedPath == null) {
            return null;
        }

        if (trimmedPath.startsWith("?")) {
            trimmedPath = trimmedPath.substring(1);
        }

        if (trimmedPath.endsWith("?") || trimmedPath.endsWith("#")) {
            trimmedPath = trimmedPath.substring(0, trimmedPath.length() - 1);
        }

        return trimmedPath;
    }

    /**
     * Trims an RFC-3968 encoded URL query component. If the given query is null or empty then null is returned. If the
     * given query starts with '?' then it is removed. If the given query ends with '#' then it is removed.
     * 
     * @param query query to trim
     * 
     * @return the trimmed query or null
     */
    @Nullable public static String trimOrNullQuery(@Nullable final String query) {
        String trimmedQuery = StringSupport.trimOrNull(query);
        if (trimmedQuery == null) {
            return null;
        }

        if (trimmedQuery.startsWith("?")) {
            trimmedQuery = trimmedQuery.substring(1);
        }

        if (trimmedQuery.endsWith("#")) {
            trimmedQuery = trimmedQuery.substring(0, trimmedQuery.length() - 1);
        }

        return trimmedQuery;
    }

    /**
     * Trims an RFC-3968 encoded URL fragment component. If the given fragment is null or empty then null is returned.
     * If the given fragment starts with '#' then it is removed.
     * 
     * @param fragment fragment to trim
     * 
     * @return the trimmed fragment or null
     */
    @Nullable public static String trimOrNullFragment(@Nullable final String fragment) {
        String trimmedFragment = StringSupport.trimOrNull(fragment);
        if (trimmedFragment == null) {
            return null;
        }

        if (trimmedFragment.startsWith("#")) {
            trimmedFragment = trimmedFragment.substring(1);
        }

        return trimmedFragment;
    }
    

    /**
     * Perform URL decoding on the given string.
     * 
     * @param value the string to decode
     * @param charset character encoding to apply
     * 
     * @return the decoded string
     * 
     * @since 9.2.0
     */
    @Nullable public static String doURLDecode(@Nullable final String value, @Nonnull final Charset charset) {
        if (value == null) {
            return null;
        }

        return URLDecoder.decode(value, charset);
    }

    /**
     * Perform URL decoding on the given string using UTF-8 as the encoding.
     * 
     * @param value the string to decode
     * 
     * @return the decoded string
     */
    @Nullable public static String doURLDecode(@Nullable final String value) {
        return doURLDecode(value, StandardCharsets.UTF_8);
    }

    /**
     * Perform URL encoding on the given string appropriate for form or query string parameters.
     * 
     * <p>This method is <strong>not</strong> appropriate for the encoding of data for other
     * parts of a URL such as a path or fragment.</p>
     * 
     * <p>Consider using Guava's UrlEscapers class for any future uses for this functionality.</p>
     * 
     * @param value the string to encode
     * @return the encoded string
     * 
     * @deprecated
     */
    @Deprecated
    @Nullable public static String doURLEncode(@Nullable final String value) {
        if (value == null) {
            return null;
        }

        return URLEncoder.encode(value, StandardCharsets.UTF_8);
    }

}