/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.httpclient;

import java.io.Closeable;
import java.io.IOException;
import java.time.Duration;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.hc.client5.http.classic.HttpClient;
import org.apache.hc.client5.http.impl.classic.RequestFailedException;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.core5.concurrent.Cancellable;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.apache.hc.core5.http.ClassicHttpResponse;
import org.apache.hc.core5.http.HttpHost;
import org.apache.hc.core5.http.protocol.HttpContext;
import org.apache.hc.core5.io.CloseMode;
import org.apache.hc.core5.io.ModalCloseable;
import org.slf4j.Logger;

import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;

/**
 * An {@link HttpClient} implementation which implements an overall request timeout for a wrapped client instance.
 * 
 * <p>
 * The request timeout {@link Duration} may be supplied as either a statically-configured value on the client instance,
 * or via the passed {@link HttpContext} in the attribute {@link HttpClientSupport#CONTEXT_KEY_REQUEST_TIMEOUT}.
 * A non-null context value will override the static client value.
 * </p>
 * 
 * <p>
 * Requests that exceed the effective timeout will throw {@link RequestTimeoutExceededException}.
 * </p>
 */
public class RequestTimeLimitingHttpClient extends AbstractHttpClient {
    
    /** Logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(RequestTimeLimitingHttpClient.class);
    
    /** The wrapped HttpClient instance. */
    @Nonnull private HttpClient httpClient;
    
    /** Executor service which implements the timeout handling. */
    @Nonnull private ScheduledExecutorService executorService;

    /** Client-level request timeout. */
    @Nullable private Duration timeout;
    
    /**
     * Constructor.
     *
     * @param client the wrapped HttpClient instance
     * @param executor executor service which implements the timeout handling
     * @param requestTimeout client-level request timeout
     */
    public RequestTimeLimitingHttpClient(@Nonnull final HttpClient client,
            @Nonnull final ScheduledExecutorService executor,
            @Nullable Duration requestTimeout) {
        super();
        httpClient = Constraint.isNotNull(client, "HttpClient was null");
        executorService = Constraint.isNotNull(executor, "ScheduledExecutorService was null");
        timeout = requestTimeout;
    }

    /** {@inheritDoc} */
    @Override
    protected ClassicHttpResponse doExecute(@Nullable final HttpHost target,
            @Nonnull final ClassicHttpRequest request,
            @Nullable final HttpContext context) throws IOException {
        
        ScheduledFuture<?> future = null;
        Duration effectiveTimeout = null;
        try {
            if (request instanceof Cancellable cancellableRequest) {
                effectiveTimeout = resolveEffectiveTimeout(context);
                if (effectiveTimeout != null) {
                    log.debug("Scheduling request timeout of duration: {}", effectiveTimeout);
                    future = executorService.schedule(cancellableRequest::cancel,
                            effectiveTimeout.toMillis(), TimeUnit.MILLISECONDS);
                }
            }
            return httpClient.executeOpen(target, request, context);
        } catch (RequestFailedException e) {
            // If we can match on the message, throw a nicer exception. But if it doesn't match, not a big deal,
            // we'll just get the original RequestFailedException.
            final String message = e.getMessage();
            if (message.contains("aborted") || message.contains("cancelled")) {
                throw new RequestTimeoutExceededException(String.format("Request to '%s'exceeded timeout '%s'",
                        request.getRequestUri(), effectiveTimeout), e);
            }
            throw e;
        } finally {
            if (future != null) {
                future.cancel(true);
            }
        }
    }
    
    /**
     * Resolve the effective request timeout to use for the current request.
     * 
     * @param context the client context
     * @return the effective timeout, may be null
     */
    @Nullable protected Duration resolveEffectiveTimeout(@Nullable final HttpContext context) {
        if (context != null) {
            final HttpClientContext clientContext = HttpClientContext.adapt(context);
            final Duration contextTimeout = clientContext.getAttribute(HttpClientSupport.CONTEXT_KEY_REQUEST_TIMEOUT,
                    Duration.class);
            if (contextTimeout != null) {
                log.debug("Resolved effective request timeout from client context: {}", contextTimeout);
                return contextTimeout;
            }
        }
        if (timeout != null) {
            log.debug("Resolved effective request timeout from statically-configured client value: {}", timeout);
        } else {
            log.debug("No effective request timeout was resolved");
        }
        return timeout;
    }

    /** {@inheritDoc} */
    @Override
    public void close() throws IOException {
        // Use try/finally here just in case the client #close() throws a RuntimeException or Error
        try {
            if (Closeable.class.isInstance(httpClient)) {
                Closeable.class.cast(httpClient).close();
            }
        } finally {
            executorService.shutdownNow();
        }
    }
    
    /** {@inheritDoc} */
    @Override
    public void close(final CloseMode closeMode) {
        // Use try/finally here just in case the client #close() throws a RuntimeException or Error
        try {
            if (ModalCloseable.class.isInstance(httpClient)) {
                ModalCloseable.class.cast(httpClient).close(closeMode);
            }
        } finally {
            executorService.shutdownNow();
        }
    }

}
