/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.totp.impl;

import java.util.function.Consumer;
import java.util.function.Function;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.security.auth.Subject;
import javax.security.auth.login.LoginException;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.authn.AbstractCredentialValidator;
import net.shibboleth.idp.authn.AuthnEventIds;
import net.shibboleth.idp.authn.CredentialValidator;
import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.authn.context.SubjectCanonicalizationContext;
import net.shibboleth.idp.authn.principal.TOTPPrincipal;
import net.shibboleth.idp.plugin.authn.totp.context.TOTPContext;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * A {@link CredentialValidator} that checks for a {@link TOTPContext}.
 */
public class TOTPCredentialValidator extends AbstractCredentialValidator {

    /** Default prefix for metrics. */
    @Nonnull @NotEmpty private static final String DEFAULT_METRIC_NAME = "net.shibboleth.idp.authn.totp"; 

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(TOTPCredentialValidator.class);

    /** Lookup strategy for TOTP context. */
    @Nonnull private Function<AuthenticationContext,TOTPContext> totpContextLookupStrategy;
    
    /** TOTP implementation. */
    @NonnullAfterInit private TOTPAuthenticator authenticator;
    
    /** Source of token seeds. */
    @NonnullAfterInit private Consumer<ProfileRequestContext> seedSource;
        
    /** A regular expression to apply for acceptance testing. */
    @Nullable private Pattern matchExpression;
    
    /** Constructor. */
    public TOTPCredentialValidator() {
        totpContextLookupStrategy = new ChildContextLookup<>(TOTPContext.class);
    }
        
    /**
     * Set the lookup strategy to locate the {@link TOTPContext}.
     * 
     * @param strategy lookup strategy
     */
    public void setTOTPContextLookupStrategy(
            @Nonnull final Function<AuthenticationContext,TOTPContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        totpContextLookupStrategy = Constraint.isNotNull(strategy, "TOTPContext lookup strategy cannot be null");
    }
    
    /**
     * Set TOTP implementation to use.
     * 
     * @param impl TOTP implementation
     */
    public void setAuthenticator(@Nonnull final TOTPAuthenticator impl) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        authenticator = Constraint.isNotNull(impl, "TOTPAuthenticator cannot be null");
    }
    
    /**
     * Set source of token seeds.
     * 
     * @param source seed source
     */
    public void setSeedSource(@Nonnull final Consumer<ProfileRequestContext> source) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        seedSource = Constraint.isNotNull(source, "Token seed source cannot be null");
    }
    
    /**
     * Set a matching expression to apply to the username for acceptance. 
     * 
     * @param expression a matching expression
     */
    public void setMatchExpression(@Nullable final Pattern expression) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        matchExpression = expression;
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (seedSource == null) {
            throw new ComponentInitializationException("Token seed source cannot be null");
        }
    }

// Checkstyle: CyclomaticComplexity OFF
    /** {@inheritDoc} */
    @Override
    protected Subject doValidate(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext,
            @Nullable final WarningHandler warningHandler,
            @Nullable final ErrorHandler errorHandler) throws Exception {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        
        final TOTPContext totpContext = totpContextLookupStrategy.apply(authenticationContext);
        if (totpContext == null) {
            log.info("{} No TOTPContext available", getLogPrefix());
            if (errorHandler != null) {
                errorHandler.handleError(profileRequestContext, authenticationContext, AuthnEventIds.NO_CREDENTIALS,
                        AuthnEventIds.NO_CREDENTIALS);
            }
            throw new LoginException(AuthnEventIds.NO_CREDENTIALS);
        } else if (totpContext.getUsername() == null) {
            log.info("{} No username available within TOTPContext", getLogPrefix());
            if (errorHandler != null) {
                errorHandler.handleError(profileRequestContext, authenticationContext, AuthnEventIds.UNKNOWN_USERNAME,
                        AuthnEventIds.UNKNOWN_USERNAME);
            }
            throw new LoginException(AuthnEventIds.NO_CREDENTIALS);
        } else if (totpContext.getTokenCode() == null) {
            log.info("{} No tokencode available within TOTPContext", getLogPrefix());
            if (errorHandler != null) {
                errorHandler.handleError(profileRequestContext, authenticationContext, AuthnEventIds.NO_CREDENTIALS,
                        AuthnEventIds.NO_CREDENTIALS);
            }
            throw new LoginException(AuthnEventIds.NO_CREDENTIALS);
        }
        
        if (totpContext.getTokenSeeds().isEmpty()) {
            // Resolve seeds.
            seedSource.accept(profileRequestContext);
            
            if (totpContext.getTokenSeeds().isEmpty()) {
                log.info("{} No seeds were obtained for user '{}'", getLogPrefix(), totpContext.getUsername());
                if (errorHandler != null) {
                    errorHandler.handleError(profileRequestContext, authenticationContext,
                            AuthnEventIds.INVALID_CREDENTIALS, AuthnEventIds.INVALID_CREDENTIALS);
                }
                throw new LoginException(AuthnEventIds.INVALID_CREDENTIALS);
            }
        }
        
        if (matchExpression != null && !matchExpression.matcher(totpContext.getUsername()).matches()) {
            log.debug("{} Username '{}' did not match expression", getLogPrefix(), totpContext.getUsername());
            return null;
        }
                
        log.debug("{} Attempting to authenticate token code for '{}' ", getLogPrefix(), totpContext.getUsername());
        
        try {
            if (totpContext.getTokenSeeds().stream().anyMatch(
                    seed -> authenticator.validate(seed, totpContext.getTokenCode()))) {
                log.info("{} Login by '{}' succeeded", getLogPrefix(), totpContext.getUsername());
                return populateSubject(new Subject(), profileRequestContext, totpContext);
            }
            
            throw new LoginException(AuthnEventIds.INVALID_CREDENTIALS);
        } catch (final Exception e) {
            log.info("{} Login by '{}' failed", getLogPrefix(), totpContext.getUsername(), e);
            if (errorHandler != null) { 
                errorHandler.handleError(profileRequestContext, authenticationContext, e,
                        AuthnEventIds.INVALID_CREDENTIALS);
            }
            throw e;
        }
    }
// Checkstyle: CyclomaticComplexity ON
    

    /**
     * Decorate the subject with "standard" content from the validation.
     * 
     * @param subject the subject being returned
     * @param profileRequestContext current profile request context
     * @param totpContext the TOTP context being validated
     * 
     * @return the decorated subject
     */
    @Nonnull protected Subject populateSubject(@Nonnull final Subject subject,
            @Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final TOTPContext totpContext) {
        subject.getPrincipals().add(new TOTPPrincipal(totpContext.getUsername()));
        
        // Bypass c14n. We already operate on a canonical name, so just re-confirm it.
        profileRequestContext.getSubcontext(SubjectCanonicalizationContext.class, true).setPrincipalName(
                totpContext.getUsername());
        
        return super.populateSubject(subject);
    }
    
}