/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.totp.impl;

import java.util.Collection;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.Multimap;

import net.shibboleth.idp.plugin.authn.totp.context.TOTPContext;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.ThreadSafeAfterInit;
import net.shibboleth.utilities.java.support.codec.Base32Support;
import net.shibboleth.utilities.java.support.codec.Base64Support;
import net.shibboleth.utilities.java.support.codec.DecodingException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

/**
 * Token seed source implementation that returns statically defined values.
 */
@ThreadSafeAfterInit
public class StaticSeedSource extends AbstractSeedSource {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(StaticSeedSource.class);
    
    /** Token seeds. */
    @Nonnull @NonnullElements private Multimap<String,byte[]> tokenSeeds;
    
    /** Constructor. */
    public StaticSeedSource() {
        tokenSeeds = LinkedListMultimap.create();
    }
    
    /**
     * Set static token seeds, keyed by username.
     * 
     * @param seeds the seeds
     */
    public void setSeeds(@Nonnull @NonnullElements final Map<String,Collection<String>> seeds) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        Constraint.isNotNull(seeds, "Seed map cannot be null");
        
        seeds.forEach((k,v) -> {
            final String user = StringSupport.trimOrNull(k);
            if (user != null) {
                StringSupport.normalizeStringCollection(v).forEach(s -> {
                    try {
                        switch (getEncoding()) {
                            case BASE32:
                                tokenSeeds.get(user).add(Base32Support.decode(s));
                                break;
                                
                            case BASE64:
                                tokenSeeds.get(user).add(Base64Support.decode(s));
                                break;
                                
                            default:
                                throw new DecodingException("Unknown encoding type");
                        }
                    } catch (final DecodingException e) {
                        log.error("Unable to decode seed value", e);
                    }
                });
            }
        });
    }
    
    /** {@inheritDoc} */
    public void accept(@Nullable final ProfileRequestContext input) {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        
        final TOTPContext totp = getTOTPContextLookupStrategy().apply(input);
        if (totp != null) {
            if (totp.getUsername() != null) {
                totp.getTokenSeeds().addAll(tokenSeeds.get(totp.getUsername()));
            } else {
                log.warn("TOTPContext did not contain a username");
            }
        } else {
            log.warn("Unable to locate TOTPContext");
        }
    }

}