/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.totp.impl;

import java.security.GeneralSecurityException;
import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.annotation.constraint.ThreadSafeAfterInit;
import net.shibboleth.utilities.java.support.codec.Base32Support;
import net.shibboleth.utilities.java.support.codec.Base64Support;
import net.shibboleth.utilities.java.support.codec.DecodingException;
import net.shibboleth.utilities.java.support.codec.EncodingException;
import net.shibboleth.utilities.java.support.component.AbstractInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.net.UrlEscapers;
import com.warrenstrange.googleauth.GoogleAuthenticator;
import com.warrenstrange.googleauth.GoogleAuthenticatorConfig;
import com.warrenstrange.googleauth.GoogleAuthenticatorKey;

/**
 * A TOTP implementation based on Google Authenticator code.
 */
@ThreadSafeAfterInit
public class GoogleTOTPAuthenticator extends AbstractInitializableComponent implements TOTPAuthenticator {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(GoogleAuthenticator.class);
    
    /** Google Authenticator config. **/
    @NonnullAfterInit private GoogleAuthenticatorConfig authconfig;

    /** The implementation. */
    @NonnullAfterInit private GoogleAuthenticator authenticator;
    
    /**
     * Set the {@link GoogleAuthenticatorConfig} to use.
     * 
     * @param config configuration to use
     */
    public void setGoogleAuthenticatorConfig(@Nonnull final GoogleAuthenticatorConfig config) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        authconfig = Constraint.isNotNull(config, "GoogleAuthenticator cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (authconfig == null) {
            authconfig = new GoogleAuthenticatorConfig();
        }
        authenticator = new GoogleAuthenticator(authconfig);
    }

    /** {@inheritDoc} */
    public TOTPCredential createCredential(@Nullable @NotEmpty final String issuer,
            @Nullable @NotEmpty final String accountName) throws GeneralSecurityException {
        
        final byte[] secret;
        final String encodedSecret;
        final GoogleAuthenticatorKey cred = authenticator.createCredentials();
        
        final String trimmedIssuer = StringSupport.trimOrNull(issuer);
        final String trimmedName = StringSupport.trimOrNull(accountName);
        
        try {
            switch (authconfig.getKeyRepresentation()) {
                case BASE32:
                    secret = Base32Support.decode(cred.getKey());
                    encodedSecret = cred.getKey();
                    break;
                    
                case BASE64:
                    secret = Base64Support.decode(cred.getKey());
                    encodedSecret = Base32Support.encode(secret, false);
                    break;
                    
                default:
                    throw new DecodingException("Unknown key representation type");
            }
            
            return new TOTPCredential() {
                public byte[] getKey() {
                    return secret;
                }
                
                public String getTOTPURL() {
                    final String label;
                    if (trimmedName != null) {
                        if (trimmedIssuer != null) {
                            label = trimmedIssuer + ":" + trimmedName;
                        } else {
                            label = trimmedName;
                        }
                        
                    } else {
                        label = null;
                    }
                    
                    final StringBuilder url = new StringBuilder();
                    url.append("otpauth://totp/")
                        .append(UrlEscapers.urlPathSegmentEscaper().escape(label))
                        .append("?secret=")
                        .append(UrlEscapers.urlFormParameterEscaper().escape(encodedSecret));
                    if (trimmedIssuer != null) {
                        url.append("&issuer=").append(UrlEscapers.urlFormParameterEscaper().escape(issuer));
                    }
                    return url.toString();
                }
    
                public Collection<Integer> getScratchCodes() {
                    return cred.getScratchCodes();
                }
            };
        } catch (final EncodingException|DecodingException e) {
            throw new GeneralSecurityException(e);
        }
    }

    /** {@inheritDoc} */
    public boolean validate(@Nonnull @NotEmpty final byte[] secret, final int code) {
        
        final String encodedSecret;
        
        try {
            switch(authconfig.getKeyRepresentation()) {
                case BASE32:
                    encodedSecret = Base32Support.encode(secret, false);
                    break;
                    
                case BASE64:
                    encodedSecret = Base64Support.encode(secret, false);
                    break;
                    
                default:
                    throw new EncodingException("Unknown key representation type");
            }
        } catch (final EncodingException e) {
            return false;
        }
        
        return authenticator.authorize(encodedSecret, code);
    }

}