/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.totp.impl;

import java.util.function.Consumer;
import java.util.function.Function;

import javax.annotation.Nonnull;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.plugin.authn.totp.context.TOTPContext;
import net.shibboleth.utilities.java.support.annotation.constraint.ThreadSafeAfterInit;
import net.shibboleth.utilities.java.support.component.AbstractInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;

/**
 * Base class for token seed source implementations.
 */
@ThreadSafeAfterInit
public abstract class AbstractSeedSource extends AbstractInitializableComponent
        implements Consumer<ProfileRequestContext> {

    /** Seed encoding types. */
    public enum Encoding {
        /** Use Base64 encoding. */
        BASE64,
        
        /** Use Base32 encoding. */
        BASE32,
    };

    /** Lookup strategy for {@link TOTPContext}. */
    @Nonnull private Function<ProfileRequestContext,TOTPContext> totpContextLookupStrategy;

    /** The encoding to reverse for the seed. */
    @Nonnull private Encoding encoding;

    /** Constructor. */
    public AbstractSeedSource() {
        // PRC -> AuthenticationContext -> TOTPContext
        totpContextLookupStrategy = FunctionSupport.compose(new ChildContextLookup<>(TOTPContext.class),
                new ChildContextLookup<>(AuthenticationContext.class));
        
        encoding = Encoding.BASE32;

    }
    
    /**
     * Get the encoding of the seeds once they're resolved.
     * 
     * <p>Defaults to {@link Encoding#BASE32}.</p>
     * 
     * @return encoding
     */
    @Nonnull public Encoding getEncoding() {
        return encoding;
    }
    
    /**
     * Set the encoding of the seeds once they're resolved.
     * 
     * @param enc encoding
     */
    public void setEncoding(@Nonnull final Encoding enc) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        encoding = Constraint.isNotNull(enc, "Encoding cannot be null");
    }

    /**
     * Get the {@link TOTPContext} lookup strategy.
     * 
     * @return lookup strategy
     */
    @Nonnull public Function<ProfileRequestContext,TOTPContext> getTOTPContextLookupStrategy() {
        return totpContextLookupStrategy;
    }

    /**
     * Set the {@link TOTPContext} lookup strategy.
     * 
     * @param strategy lookup strategy
     */
    public void setTOTPContextLookupStrategy(@Nonnull final Function<ProfileRequestContext,TOTPContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        totpContextLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

}