/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.totp.impl;

import java.security.GeneralSecurityException;
import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.annotation.constraint.Unmodifiable;

/**
 * Interface to a TOTP implementation's primitives.
 */
public interface TOTPAuthenticator {

    /**
     * Generate a new credential.
     * 
     * @param issuer TOTP credential issuer
     * @param accountName TOTP account holder
     * 
     * @return a new credential
     * 
     * @throws GeneralSecurityException if unable to generate the credential
     */
    @Nonnull TOTPCredential createCredential(@Nullable @NotEmpty final String issuer,
            @Nullable @NotEmpty final String accountName) throws GeneralSecurityException;
    
    /**
     * Validate a secret and code.
     * 
     * @param secret token secret/seed, unencoded
     * @param code token code to validate
     * 
     * @return true iff the code is valid
     */
    boolean validate(@Nonnull @NotEmpty final byte[] secret, final int code);
    
    /** Interface to a TOTP credential. */
    public interface TOTPCredential {
        
        /**
         * Get the secret key.
         *
         * @return the secret key
         */
        @Nonnull @NotEmpty byte[] getKey();

        /**
         * Get the TOTP URL associated with the credential.
         * 
         * @return TOTP URL
         */
        @Nonnull @NotEmpty String getTOTPURL();
        
        /**
         * Get the list of scratch codes.
         *
         * @return the list of scratch codes
         */
        @Nonnull @NonnullElements @Unmodifiable @NotLive Collection<Integer> getScratchCodes();
    }

}