/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.totp.impl;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;

import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An action that derives the TOTP code from a form parameter.
 */
public class ExtractTOTPFromFormRequest extends AbstractTOTPExtractionAction {

    /** Default token code field name. */
    @Nonnull @NotEmpty public static final String DEFAULT_FIELD_NAME = "tokencode";
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(ExtractTOTPFromFormRequest.class);
    
    /** Name of header. */
    @NonnullAfterInit @NotEmpty private String fieldName;
    
    /** Constructor. */
    public ExtractTOTPFromFormRequest() {
        fieldName = DEFAULT_FIELD_NAME;
    }
    
    /**
     * Set the name of the field to examine.
     * 
     * @param field field name
     */
    public void setFieldName(@Nonnull @NotEmpty final String field) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        fieldName = Constraint.isNotNull(StringSupport.trimOrNull(field), "Field name cannot be null or empty");
    }

    /** {@inheritDoc} */
    @Override
    @Nullable protected String extractCode(@Nonnull final HttpServletRequest httpRequest) {
        return httpRequest.getParameter(fieldName);
    }
    
}