/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.impl;

import java.text.ParseException;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTClaimsSet;

import net.shibboleth.idp.authn.AuthnEventIds;
import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.plugin.authn.duo.AbstractDuoAuthenticationAction;
import net.shibboleth.idp.plugin.authn.duo.DuoException;
import net.shibboleth.idp.plugin.authn.duo.context.DuoOIDCAuthenticationContext;
import net.shibboleth.oidc.jwt.claims.ClaimsValidator;
import net.shibboleth.oidc.jwt.claims.JWTValidationException;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Action that validates the claims of the Duo id_token using the supplied {@link ClaimsValidator}. 
 * The verifier <b>must</b> be thread-safe and validate the claims set against the OpenID Connect
 * core 1.0 section 3.1.3.7 specification and those required by Duo. 
 * 
 * @pre
 * 
 *      <pre>
 *      ProfileRequestContext.getSubcontext(AuthenticationContext.class, false) != null
 *      </pre>
 * 
 * @pre
 * 
 *      <pre>
 *      AuthenticationContext.getSubcontext(DuoOIDCAuthenticationContext.class, false) != null
 *      </pre>
 * 
 * @pre
 * 
 *      <pre>
 *      DuoOIDCAuthenticationContext.getAuthToken() != null
 *      </pre>
 * @pre
 * 
 *      <pre>
 *      DuoOIDCAuthenticationContext.getIntegration() != null
 *      </pre>
 * 
 * @event {@link org.opensaml.profile.action.EventIds#PROCEED_EVENT_ID}
 * @event {@link net.shibboleth.idp.authn.AuthnEventIds#AUTHN_EXCEPTION}
 * @event {@link net.shibboleth.idp.authn.AuthnEventIds#NO_CREDENTIALS}
 */
public class ValidateTokenClaims extends AbstractDuoAuthenticationAction {
   
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(ValidateTokenClaims.class);
    
    /** The parsed claimset. */
    @Nullable private JWTClaimsSet claimsSet;
    
    /** A cleanup hook to execute after either a successful or unsuccessful claims validation. */
    @Nullable private Consumer<ProfileRequestContext> cleanupHook;
    
    /** The JWT claims validator used to verify the claimsset.*/
    @NonnullAfterInit private ClaimsValidator claimsValidator;
    
    /** {@inheritDoc} */
    @Override protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        if (claimsValidator ==  null) {
            throw new ComponentInitializationException("Duo ClaimSet Validator cannot be null");
        }
    }

    /**
     * Set the cleanup hook to execute after either a successful or unsuccessful claims validation.
     * 
     * @param hook cleanup hook
     * 
     */
    public void setCleanupHook(@Nullable final Consumer<ProfileRequestContext> hook) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        cleanupHook = hook;
    }
    
    /**
     * Set the JWT claims verifier to use.
     * 
     * @param validator the claims validator.
     */
    public void setClaimsValidator(
            @Nonnull final ClaimsValidator validator) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        claimsValidator = Constraint.isNotNull(validator, "Claims validator cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext,
            @Nonnull final DuoOIDCAuthenticationContext duoContext) {

        final JWT token = duoContext.getAuthToken();
        if (token == null) {
            log.error("{} Duo 2FA token is not available", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.INVALID_AUTHN_CTX);
            return false;
        }
        try {
            //parse the claimset here, so parsing only has to happen once, and we fail fast on error (e.g. bad JSON)
            claimsSet = token.getJWTClaimsSet();
            if (claimsSet == null) {
                throw new DuoException("Duo JWT ClaimsSet is null");
            }
        } catch (final ParseException | DuoException e) {
            log.error("{} Claimset of Duo 2FA token is not available", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.INVALID_AUTHN_CTX);
            return false;
        }        
        return true;
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext,
            @Nonnull final DuoOIDCAuthenticationContext duoContext) {

        log.debug("{} Validating token claims for subject '{}'",getLogPrefix(),claimsSet.getSubject());
         
        try {
            claimsValidator.validate(claimsSet,profileRequestContext);
            if (cleanupHook != null) {
                cleanupHook.accept(profileRequestContext);
            }
        } catch (final JWTValidationException e) {
            log.error("{} Token verification failed for subject '{}'", getLogPrefix(),claimsSet.getSubject(),e);
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.NO_CREDENTIALS);
            if (cleanupHook != null) {
                cleanupHook.accept(profileRequestContext);
            }
            return;
        }
        //fine.
        log.debug("{} Token claims are valid for subject '{}'",getLogPrefix(),claimsSet.getSubject());
    }
    
    /**
     * A cleanup hook that removes the 'nonce' parameter from the {@link DuoOIDCAuthenticationContext} so
     * it could not be reused.
     */
    public static class DuoOIDAuthenticationContextCleanupHook implements Consumer<ProfileRequestContext> {

        /** {@inheritDoc} */
        @Override
        public void accept(@Nullable final ProfileRequestContext input) {
            if (input != null) {
                final AuthenticationContext authnCtx = input.getSubcontext(AuthenticationContext.class);
                if (authnCtx != null) {
                    final DuoOIDCAuthenticationContext duoCtx = authnCtx.getSubcontext(
                            DuoOIDCAuthenticationContext.class);
                    if (duoCtx != null) {
                        duoCtx.setNonce(null);
                    }
                }
            }
        }
    }
}
