/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.impl;

import java.security.SecureRandom;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;

import net.shibboleth.idp.plugin.authn.duo.DuoException;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Helper methods for Duo 2FA. 
 */
@ThreadSafe
public final class DuoSupport {
    
    
    /** Private Constructor. */
    private DuoSupport() {

    }
    
   
    /**
     * Generates a random identifier to be used as a nonce.
     *  
     * @param length the length of the parameter, minimum allowed is 22.
     * 
     * @return the randomly generated nonce value.
     */
    @Nonnull static String generateNonce(@Nonnull final Integer length) {
        Constraint.isGreaterThan(22, length, "Nonce must be at least 22 characters");
        final SecureRandom secureRandom = new SecureRandom();
        final StringBuilder sb = new StringBuilder();
        while(sb.length() < length){
            sb.append(Integer.toHexString(secureRandom.nextInt()));
        }
        return sb.toString().substring(0, length);
    }
        
    
    /**
     * <p>Generate a state parameter from a nonce component and an execution key component.</p>
     * 
     * <p>The nonce is separated from the key by a dot e.g. {@literal <nonce>.<keyHex>}.</p>
     * 
     *  <p>The nonce is assumed to be already encoded in its transmission format e.g. Hex. The key is
     *  hex encoded before it is combined with the nonce. The result is assumed URL encoded e.g. inside
     *  the allowed set of URI characters or, no character in the state is from the URI reserved set.</p>
     * 
     * @param nonce the nonce component. 
     * @param key the key component. The key is hex encoded before it is added to the generated state.
     * 
     * @return the combined state component.
     */
    @Nonnull static String generateState(@Nonnull final String nonce, @Nonnull final String key) {
        Constraint.isNotNull(nonce, "Nonce Hex key can not be null");
        Constraint.isNotNull(key, "Webflow execution key can not be null");
        
        final String keyHex = Hex.encodeHexString(key.getBytes());
        return nonce+"."+keyHex;
    }
    
    /**
     * Extract the key component from the state. The key is hex encoded and separated from the
     * nonce value by a dot. 
     * 
     * @param state the state which contains both the nonce and the key dot separated.
     * 
     * @return the key extracted from the state and hex decoded.
     * 
     * @throws DuoException if the key component can not be found, or hex decoding fails.
     */
    @Nonnull static String extractKeyFromState(@Nonnull final String state) throws DuoException{
        Constraint.isNotNull(state, "State can not be null");
        
        final String[] stateSplit = state.split("\\.");
        if (stateSplit.length!=2) {
            throw new DuoException("State does not contain the key component");
        }
        final String hexKey = stateSplit[1];
        try {
            //should we check it is sensible?
            return new String(Hex.decodeHex(hexKey));
        } catch (final DecoderException e) {
            throw new DuoException("Can not hex decode key",e);
        }
        
    }

    /**
     * Extract the nonce component from the state. The nonce is separated from the key
     * by a dot, and is assumed to be the first value of the pair.
     * 
     * @param state the state which contains both the nonce and the key dot separated.
     * 
     * @return the nonce extracted from the state.
     * 
     * @throws DuoException if the nonce component can not be found.
     */
    @Nonnull static String extractNonceFromState(@Nonnull final String state) throws DuoException{
        Constraint.isNotNull(state, "State can not be null");

        final String[] stateSplit = state.split("\\.");
        if (stateSplit.length!=2) {
            throw new DuoException("State does not contain the nonce component");
        }
        return stateSplit[0];
    }

}
