/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.audit.impl;

import java.text.ParseException;
import java.util.Map;

import javax.annotation.Nonnull;

import com.nimbusds.jwt.JWTClaimsSet;

import net.shibboleth.idp.plugin.authn.duo.DuoOIDCAuthAPI;
import net.shibboleth.idp.plugin.authn.duo.audit.AbstractDuoTokenClaimsAuditExtractor;
import net.shibboleth.idp.plugin.authn.duo.context.DuoOIDCAuthenticationContext;

/** An audit extractor to extract the Duo transaction ID from the Duo token response.*/
public class DuoTransactionIdAuditExtractor extends AbstractDuoTokenClaimsAuditExtractor<String> {

    @Override
    protected String doLookup(
            @Nonnull final JWTClaimsSet claimsSet, @Nonnull final DuoOIDCAuthenticationContext context) {
        String txid = null;
        try {
            final Map<String, Object> authnContextClaimObj = 
                    claimsSet.getJSONObjectClaim(DuoOIDCAuthAPI.DUO_AUTH_CONTEXT_JSON_OBJECT);
            
            if (authnContextClaimObj != null) {         
                final Object txidObj = 
                        authnContextClaimObj.get(DuoOIDCAuthAPI.DUO_TXID_JSON_OBJECT);
                if (txidObj instanceof String) {
                    txid = (String)txidObj;
                }
            }
        } catch (final ParseException e) {
            // Do nothing, just return null
        }
        return txid;
    }


}
