/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.audit.impl;

import java.text.ParseException;
import java.util.Map;

import javax.annotation.Nonnull;

import com.nimbusds.jwt.JWTClaimsSet;

import net.shibboleth.idp.plugin.authn.duo.DuoOIDCAuthAPI;
import net.shibboleth.idp.plugin.authn.duo.audit.AbstractDuoTokenClaimsAuditExtractor;
import net.shibboleth.idp.plugin.authn.duo.context.DuoOIDCAuthenticationContext;

/** An audit extractor to extract the Duo 2FA device name from the auth_device in the Duo token response.*/
public class DuoAuthnDeviceNameAuditExtractor extends AbstractDuoTokenClaimsAuditExtractor<String> {

    @Override
    protected String doLookup(
            @Nonnull final JWTClaimsSet claimsSet, @Nonnull final DuoOIDCAuthenticationContext context) {
        String value = null;
        try {
            final Map<String, Object> authnContextClaimObj = 
                    claimsSet.getJSONObjectClaim(DuoOIDCAuthAPI.DUO_AUTH_CONTEXT_JSON_OBJECT);
            
            if (authnContextClaimObj != null) {      
                
                final Object authDeviceObj = 
                        authnContextClaimObj.get(DuoOIDCAuthAPI.DUO_AUTH_DEVICE_JSON_OBJECT);
                if (authDeviceObj instanceof Map) {
                    final Map<?, ?> mapValue = (Map<?, ?>)authDeviceObj;
                    
                    final Object valueObj = 
                            mapValue.get(DuoOIDCAuthAPI.DUO_AUTH_DEVICE_NAME_JSON_OBJECT);
                    if (valueObj instanceof String) {
                        value = (String)valueObj;
                    }
                } 
            }
        } catch (final ParseException e) {
            // Do nothing, just return null
        }
        return value;
    }


}
