/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.impl;

import java.security.Principal;
import java.text.ParseException;
import java.util.Collection;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.security.auth.Subject;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTClaimsSet;

import net.shibboleth.idp.authn.AbstractValidationAction;
import net.shibboleth.idp.authn.AuthenticationResult;
import net.shibboleth.idp.authn.AuthnEventIds;
import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.authn.context.SubjectCanonicalizationContext;
import net.shibboleth.idp.authn.duo.DuoPrincipal;
import net.shibboleth.idp.plugin.authn.duo.DuoException;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCAuthAPI;
import net.shibboleth.idp.plugin.authn.duo.context.DuoOIDCAuthenticationContext;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.component.ComponentSupport;

/**
 * A validation action that checks for a valid Duo authentication token and directly produces an
 * {@link AuthenticationResult} based on the identity described by the token.
 * 
 * @event {@link org.opensaml.profile.action.EventIds#PROCEED_EVENT_ID}
 * @event {@link AuthnEventIds#AUTHN_EXCEPTION}
 * @event {@link AuthnEventIds#INVALID_AUTHN_CTX}
 * @pre <pre>
 *      ProfileRequestContext.getSubcontext(AuthenticationContext.class, false) != null
 *      </pre>
 * 
 * @pre <pre>
 *      AuthenticationContext.getSubcontext(DuoOIDCAuthenticationContext.class, false) != null
 *      </pre>
 */
public class ValidateDuoTokenAuthenticationResult extends AbstractValidationAction{
    
    /** Class logger.*/
    @Nonnull private final Logger log = LoggerFactory.getLogger(ValidateDuoTokenAuthenticationResult.class);
       
    /** Duo authentiction context. */
    @Nullable private DuoOIDCAuthenticationContext duoContext;
    
    /** The profile request context.*/
    @Nullable private ProfileRequestContext prc;
    
    /** The parsed claimset. */
    @Nullable private JWTClaimsSet claimsSet;
    
    /** Attempted username. */
    @Nullable @NotEmpty private String username;
    
    /** Hook to map context information, often Duo factors in the Duo token, to principal collections.*/
    @Nullable private Function<ProfileRequestContext,Collection<Principal>> contextToPrincipalMappingStrategy; 
    
    /**
     * Get the context to principal mapping strategy for mapping context information into 
     * principal collections e.g. Duo factors.
     * 
     * @return the mapping hook
     */
    @Nullable public Function<ProfileRequestContext,Collection<Principal>> getContextToPrincipalMappingStrategy() {
        return contextToPrincipalMappingStrategy;
    }
    
    /**
     * Set the context to principal mapping strategy for mapping context information into 
     * principal collections e.g. Duo factors.
     * 
     * @param hook principal mapping hook
     */
    public void setContextToPrincipalMappingStrategy(@Nullable final 
            Function<ProfileRequestContext,Collection<Principal>> hook) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        contextToPrincipalMappingStrategy = hook;
    }

    
    /** {@inheritDoc} */
    @Override protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext) {

        if (!super.doPreExecute(profileRequestContext, authenticationContext)) {
            return false;
        }
        //stash the profile request context for later.
        prc = profileRequestContext;

        duoContext = authenticationContext.getSubcontext(DuoOIDCAuthenticationContext.class);
        if (duoContext == null) {
            log.error("{} No DuoAuthenticationContext available", getLogPrefix());
            handleError(profileRequestContext, authenticationContext, "No DuoAuthenticationContext context available",
                    AuthnEventIds.INVALID_AUTHN_CTX);
            recordFailure(profileRequestContext);
            return false;
        }        
        //we get username from the original context, not the duo response.
        username = duoContext.getUsername();      
        
        final JWT token = duoContext.getAuthToken();
        if (token == null) {
            log.error("{} Duo 2FA token is not available", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.INVALID_AUTHN_CTX);
            recordFailure(profileRequestContext);
            return false;
        }
        try {
            //parse the claimset here, so parsing only has to happen once, and we fail fast on error (e.g. bad JSON)
            claimsSet = token.getJWTClaimsSet();
            if (claimsSet == null) {
                throw new DuoException("Duo JWT ClaimsSet is null");
            }
        } catch (final ParseException | DuoException e) {
            log.error("{} Claimset of Duo 2FA token is not available", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.INVALID_AUTHN_CTX);
            recordFailure(profileRequestContext);
            return false;
        }
        
        return true;
    }

    
    /** {@inheritDoc} */
    @Override protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext) {    
        
                 
        Map<String, Object> authStatusObject = null;
        try {
            authStatusObject = claimsSet.getJSONObjectClaim(DuoOIDCAuthAPI.DUO_AUTH_RESULT_JSON_OBJECT);
            if (authStatusObject == null) {
                throw new DuoException("Authentication result object is null");
            }
        } catch (final ParseException | DuoException e) {
            log.error("{} Duo 2FA access failed for '{}', auth_result missing",getLogPrefix(), username);
            handleError(profileRequestContext, authenticationContext,AuthnEventIds.AUTHN_EXCEPTION, 
                    AuthnEventIds.AUTHN_EXCEPTION);
            recordFailure(profileRequestContext);
            return;
        }
        
        final Object statusObj = authStatusObject.get(DuoOIDCAuthAPI.DUO_AUTH_RESULT_STATUS_JSON_OBJECT);
        final Object statusMsgObj = authStatusObject.get(DuoOIDCAuthAPI.DUO_AUTH_RESULT_STATUS_MSG_JSON_OBJECT);
        
        //instanceof includes null check
        if (statusObj instanceof String && statusMsgObj instanceof String) {
            final String authResultStatus = (String)statusObj;
            final String authResultStatusMsg = (String)statusMsgObj;
            
            if (DuoOIDCAuthAPI.DUO_AUTH_RESULT_ALLOW.equalsIgnoreCase(authResultStatus)){
                log.info("{} Duo 2FA authentication succeeded for '{}'",getLogPrefix(),duoContext.getUsername());
                recordSuccess(profileRequestContext);
                buildAuthenticationResult(profileRequestContext, authenticationContext);
                return;
            } else if (DuoOIDCAuthAPI.DUO_AUTH_RESULT_DENY.equalsIgnoreCase(authResultStatus)) {
                log.error("{} Duo 2FA failed for '{}', 2FA status '{}'",getLogPrefix(), username,
                        authResultStatusMsg);
                handleError(profileRequestContext, authenticationContext, authResultStatus,
                        AuthnEventIds.INVALID_CREDENTIALS);
                recordFailure(profileRequestContext);
                return;
            } else {        
                log.error("{} Duo 2FA access failed for '{}', unknown response", getLogPrefix(), username);
                handleError(profileRequestContext, authenticationContext,AuthnEventIds.AUTHN_EXCEPTION, 
                        AuthnEventIds.AUTHN_EXCEPTION);
                recordFailure(profileRequestContext);
                return;
            }
        } else {
            log.error("{} Duo 2FA access failed for '{}', auth_results missing",getLogPrefix(), username);
            handleError(profileRequestContext, authenticationContext,AuthnEventIds.AUTHN_EXCEPTION, 
                    AuthnEventIds.AUTHN_EXCEPTION);
            recordFailure(profileRequestContext);
            return;
        }
       
        
        
        
    }
    
    /** {@inheritDoc} */
    @Override protected Subject populateSubject(@Nonnull final Subject subject) {
        
        //Always add the custom Duo principal
        subject.getPrincipals().add(new DuoPrincipal(username));
        //Always add any principals specified on the integration
        subject.getPrincipals().addAll(duoContext.getIntegration().getSupportedPrincipals(Principal.class));
        
        //add any further principals from a function hook that can inspect the Duo response.
        //If the mapping strategy is set, the defaults should not be copied over from the flow. Hence,
        //these will be added only to those added above (this is configured in the XML).
        if (getContextToPrincipalMappingStrategy() != null) {
            final Collection<Principal> mapped = getContextToPrincipalMappingStrategy().apply(prc);
            if (mapped != null) {
                subject.getPrincipals().addAll(mapped);
                if (log.isDebugEnabled()) {
                    log.debug("{} Added mapped Principals: {}", getLogPrefix(),
                            mapped.stream().map(Principal::getName).collect(Collectors.toUnmodifiableList()));
                }
            }
        }
        
        return subject;
    }

    /** {@inheritDoc} */
    @Override protected void buildAuthenticationResult(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext) {
        super.buildAuthenticationResult(profileRequestContext, authenticationContext);

         // Bypass c14n. We already operate on a canonical name, so just re-confirm it.
        profileRequestContext.getSubcontext(SubjectCanonicalizationContext.class, true).setPrincipalName(username);
    }
    
    /**
     * A default cleanup hook that removes the {@link DuoOIDCAuthenticationContext} from the tree.
     */
    public static class DuoOIDCCleanupHook implements Consumer<ProfileRequestContext> {

        /** {@inheritDoc} */
        public void accept(@Nullable final ProfileRequestContext input) {
            if (input != null) {
                final AuthenticationContext authnCtx = input.getSubcontext(AuthenticationContext.class);
                if (authnCtx != null) {
                    final DuoOIDCAuthenticationContext duoCtx = 
                            authnCtx.getSubcontext(DuoOIDCAuthenticationContext.class);
                    if (duoCtx != null) {
                        authnCtx.removeSubcontext(duoCtx);
                    }
                }
            }
        }
    }

}
