/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.impl;

import java.util.function.Function;

import javax.annotation.Nonnull;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.authn.AbstractAuthenticationAction;
import net.shibboleth.idp.authn.AuthnEventIds;
import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.authn.duo.DuoIntegration;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCClient;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCClientRegistry;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCIntegration;
import net.shibboleth.idp.plugin.authn.duo.DuoRegistryException;
import net.shibboleth.idp.plugin.authn.duo.context.DuoOIDCAuthenticationContext;
import net.shibboleth.idp.session.context.navigate.CanonicalUsernameLookupStrategy;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;

/**
 * An action to create (or lookup) and populate the {@link DuoOIDCAuthenticationContext} 
 * with the username, chosen {@link DuoIntegration}, and {@link DuoOIDCClient} appropriate for this request. 
 * 
 * @event {@link org.opensaml.profile.action.EventIds#PROCEED_EVENT_ID}
 * @event {@link org.opensaml.profile.action.EventIds#INVALID_PROFILE_CTX}
 * @event {@link net.shibboleth.idp.authn.AuthnEventIds#NO_CREDENTIALS}
 * @event {@link net.shibboleth.idp.authn.AuthnEventIds#AUTHN_EXCEPTION}
 * @post See above.
 */
public class PopulateDuoAuthenticationContext extends AbstractAuthenticationAction {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(PopulateDuoAuthenticationContext.class);
    
    /** Strategy used to locate or create the {@link DuoOIDCAuthenticationContext} to populate. */
    @Nonnull private Function<ProfileRequestContext,DuoOIDCAuthenticationContext> duoAuthContextCreationStrategy;

    /** Lookup strategy for username to match against Duo identity. */
    @Nonnull private Function<ProfileRequestContext, String> usernameLookupStrategy;

    /** Lookup strategy for Duo integration. */
    @Nonnull private Function<ProfileRequestContext, DuoOIDCIntegration> duoIntegrationLookupStrategy;
    
    /** The registry for locating the DuoClient for the established integration.*/
    @NonnullAfterInit private DuoOIDCClientRegistry clientRegistry;

    /** Constructor.*/
    public PopulateDuoAuthenticationContext() {
        //default creates duo authentication context under authentication context.
        duoAuthContextCreationStrategy =
                new ChildContextLookup<>(DuoOIDCAuthenticationContext.class, true).
                compose(new ChildContextLookup<>(AuthenticationContext.class));
        
        usernameLookupStrategy = new CanonicalUsernameLookupStrategy();
        duoIntegrationLookupStrategy = FunctionSupport.constant(null);
    }
    
    /**
     * Set the Duo client registry.
     * 
     * @param duoRegistry the registry
     */
    public void setClientRegistry(@Nonnull final DuoOIDCClientRegistry duoRegistry) {        
        clientRegistry = Constraint.isNotNull(duoRegistry,"DuoClient registry can not be null");
    }

    /**
     * Set the lookup strategy to use for the username to match against Duo identity.
     * 
     * @param strategy lookup strategy
     */
    public void setUsernameLookupStrategy(@Nonnull final Function<ProfileRequestContext, String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        usernameLookupStrategy = Constraint.isNotNull(strategy, "Username lookup strategy cannot be null");
    }
    
    /**
     * Set the strategy used to locate the {@link DuoOIDCAuthenticationContext} to operate on.
     * 
     * @param strategy lookup strategy
     */
    public void setDuoContextCreationStrategy(
            @Nonnull final Function<ProfileRequestContext,DuoOIDCAuthenticationContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        duoAuthContextCreationStrategy = Constraint.isNotNull(strategy, "DuoAuthenticationContext"
                + " creation strategy cannot be null");
    }

    /**
     * Set DuoIntegration lookup strategy to use.
     * 
     * @param strategy lookup strategy
     */
    public void setDuoIntegrationLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, DuoOIDCIntegration> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        duoIntegrationLookupStrategy = Constraint.isNotNull(strategy, "DuoIntegration lookup strategy cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        if (clientRegistry ==  null) {
            throw new ComponentInitializationException("Duo Client Registry cannot be null");
        }
    }

    /** {@inheritDoc} */
    @Override protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext) {

        
        final DuoOIDCAuthenticationContext context = duoAuthContextCreationStrategy.apply(profileRequestContext);
        if (context == null) {
            log.error("{} Error creating DuoAuthenticationContext", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return;
        }
        
        final DuoOIDCIntegration duoIntegration = duoIntegrationLookupStrategy.apply(profileRequestContext);
        if (duoIntegration == null) {
            log.warn("{} No DuoIntegration returned by lookup strategy", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return;
        }
        context.setIntegration(duoIntegration);
        
        final String username = usernameLookupStrategy.apply(profileRequestContext);
        if (username == null) {
            log.warn("{} No principal name available to initiate a Duo 2FA request", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.NO_CREDENTIALS);
            return;
        }
        context.setUsername(username);
        
        //Configure the Duo client for the established integration
        try {
            final DuoOIDCClient client = clientRegistry.getClientOrCreate(duoIntegration);
            context.setClient(client);
        } catch (final DuoRegistryException e) {
            log.warn("{} No DuoClient established (located or created) for "
                    + "this integration", getLogPrefix(),e);
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.AUTHN_EXCEPTION);
            return;
        }
        
        log.debug("Created Duo authentication context for '{}'",username);
    }

}
