/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.impl;

import java.text.ParseException;

import javax.annotation.Nonnull;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jwt.JWT;

import net.shibboleth.idp.authn.AuthnEventIds;
import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.plugin.authn.duo.AbstractDuoAuthenticationAction;
import net.shibboleth.idp.plugin.authn.duo.DuoClientException;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCClient;
import net.shibboleth.idp.plugin.authn.duo.context.DuoOIDCAuthenticationContext;


/**
 * Action to exchange the authorization code in the Duo 2FA response for a Duo id_token that describes the result
 * of 2FA. Once obtained, adds the token to the Duo context.
 * 
 * @event {@link org.opensaml.profile.action.EventIds#PROCEED_EVENT_ID}
 * @event {@link AuthnEventIds#AUTHN_EXCEPTION}
 * @event {@link AuthnEventIds#NO_CREDENTIALS}
 * @pre <pre>ProfileRequestContext.getSubcontext(AuthenticationContext.class, false) != null and 
 * AuthenticationContext.getSubcontect(DuoOIDCAuthenticationContext.class,false)!=null</pre>
 * @post Add the Duo authentication token to the context. 
 */
public class ExchangeCodeForDuoToken extends AbstractDuoAuthenticationAction{
    
    /** Class logger.*/
    @Nonnull private final Logger log = LoggerFactory.getLogger(ExchangeCodeForDuoToken.class);
    
    
    /** {@inheritDoc} */
    @Override protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext,
            @Nonnull final DuoOIDCAuthenticationContext duoContext) { 
        
        final DuoOIDCClient client = duoContext.getClient();
        if (client == null) {
            log.error("{} Duo client is null, has the context been created correctly?",getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.AUTHN_EXCEPTION);
            return;
        }
        
        final String code = duoContext.getAuthorizationCode();        
        if (code == null) {
            log.error("{} Duo 2FA authorization code is not available in the response",getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.NO_CREDENTIALS);
            return;
        }
        final String username = duoContext.getUsername();
        if (username == null) {
            log.error("{} Username is not available in the Duo context",getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.NO_CREDENTIALS);
            return;
        }
        
        try {
            final JWT token = client.exchangeAuthorizationCodeFor2FAResult(code,username);
            if (log.isDebugEnabled()) {
                //avoid parsing claims if debug not enabled, 
                //if debug is enabled and parsing fails here, you will get different behaviour than if
                //debug is not enabled!
                log.debug("{} Duo 2FA token received for subject '{}'",getLogPrefix(),
                        token.getJWTClaimsSet().getSubject());
            }
            duoContext.setAuthToken(token);            
            //success
        } catch (final DuoClientException | ParseException e) {
            log.error("{} Unable to exchange authorisation code for 2FA result",getLogPrefix(),e);
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.AUTHN_EXCEPTION);
            return;
        }
        
    }

}
