/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.impl;

import java.io.IOException;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;

import net.shibboleth.idp.authn.ExternalAuthentication;
import net.shibboleth.idp.authn.ExternalAuthenticationException;
import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.plugin.authn.duo.DuoClientException;
import net.shibboleth.idp.plugin.authn.duo.DuoException;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCClient;
import net.shibboleth.idp.plugin.authn.duo.context.DuoOIDCAuthenticationContext;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.component.AbstractInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * 
 * <p>MVC controller for managing Duo 2FA exchanges implemented as an {@link ExternalAuthentication} 
 * mechanism.</p>
 * 
 * <p>The controller initiates the Duo OIDC authorization code grant flow and accepts the authorization 
 * code response.</p>
 *  
 */
@Controller
@RequestMapping("%{idp.duo.oidc.externalAuthnPath:/Authn/Duo/2FA}")
public class DuoOIDCAuthnController extends AbstractInitializableComponent{
    
    /** The name of the Http parameter that stores the authorisation code.*/
    @Nonnull @NotEmpty public static final String CODE_PARAMETER = "code";
    
    /** 
     * The name of the Http parameter that stores the authorisation code
     * when using the Duo WebSDK client.
     */
    @Nonnull @NotEmpty public static final String DUO_CODE_PARAMETER = "duo_code";
    
    /** The name of the Http parameter that stores the state value.*/
    @Nonnull @NotEmpty public static final String STATE_PARAMETER = "state";
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(DuoOIDCAuthnController.class);    
    
    /** Lookup strategy to locate the Duo authentication context. */
    @Nonnull private Function<ProfileRequestContext,DuoOIDCAuthenticationContext> duoContextLookupStrategy;
    
    /** Constructor. */
    public DuoOIDCAuthnController() {
        // PRC -> AC -> DuoAuthenticationContext
        duoContextLookupStrategy = new ChildContextLookup<>(DuoOIDCAuthenticationContext.class).
                compose(new ChildContextLookup<>(AuthenticationContext.class));
    }
    
    /**
     * Set Duo authentication context lookup strategy to use.
     * 
     * @param strategy lookup strategy
     */
    public void setDuoContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,DuoOIDCAuthenticationContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        duoContextLookupStrategy = Constraint.isNotNull(strategy, "DuoContext lookup strategy cannot be null");
    }
    
    /**
     * Start the Duo ODIC authorization code flow. The SWF execution key is encoded in the state parameter
     * so it can be extracted on return from Duo.
     * 
     * @param httpRequest servlet request
     * @param httpResponse servlet response
     * 
     * @throws ExternalAuthenticationException if an error occurs
     * @throws IOException if an I/O error occurs
     */
    @GetMapping("/authorize")
    public void authorizationRequest(@Nonnull final HttpServletRequest httpRequest,
            @Nonnull final HttpServletResponse httpResponse) throws ExternalAuthenticationException, IOException {
        
        final String key = ExternalAuthentication.startExternalAuthentication(httpRequest);        
        final ProfileRequestContext prc = ExternalAuthentication.getProfileRequestContext(key, httpRequest);
        
        final DuoOIDCAuthenticationContext duoContext = duoContextLookupStrategy.apply(prc);
        if (duoContext == null) {
            log.error("No Duo context to use in initiating a Duo 2FA request");
            httpRequest.setAttribute(ExternalAuthentication.AUTHENTICATION_ERROR_KEY, EventIds.INVALID_PROFILE_CTX);
            ExternalAuthentication.finishExternalAuthentication(key, httpRequest, httpResponse);
            return;
            
        }

        log.trace("Starting Duo 2FA for integration client '{}' and user '{}'",
                duoContext.getIntegration() != null ? duoContext.getIntegration().getClientId(): 
                    "none", duoContext.getUsername());
        try {
            final DuoOIDCClient client = duoContext.getClient();
            if (client == null) {
                throw new DuoClientException("Duo client is null, has the context been created correctly?");
            }            
            //generate state, stash in the context for checking on return.
            //TODO: could use a type of replay cache and storage service?
            final String nonce = DuoSupport.generateNonce(32);
            final String state = DuoSupport.generateState(nonce, key);
            //store only the nonce component as the request state. The key component is only used to resume the flow.
            duoContext.setRequestState(nonce);
            
            String authURL;
            //if an OIDC id_token nonce is supported, add it to the authz request
            if (client.getCapabilities().isSupportsNonce()) {
                final String oidcNonce = DuoSupport.generateNonce(36);
                authURL = client.createAuthUrl(duoContext.getUsername(), state, oidcNonce);
                duoContext.setNonce(oidcNonce);
               
            } else {
                authURL = client.createAuthUrl(duoContext.getUsername(), state, null);
            }

            httpResponse.sendRedirect(authURL);

        } catch (final DuoClientException e) {
            httpRequest.setAttribute(ExternalAuthentication.AUTHENTICATION_EXCEPTION_KEY, e);
            ExternalAuthentication.finishExternalAuthentication(key, httpRequest, httpResponse);
        }
              
    }

    /**
     * The redirect_uri endpoint for accepting an authorization code and resuming the flow execution.
     * 
     * @param httpRequest servlet request
     * @param httpResponse servlet response
     * 
     * @throws ExternalAuthenticationException if an error occurs
     * @throws IOException if an I/O error occurs
     */
    @GetMapping("/duo-callback")
    public void authorizationCallback(@Nonnull final HttpServletRequest httpRequest,
            @Nonnull final HttpServletResponse httpResponse) throws ExternalAuthenticationException, IOException {
        
        final String code = httpRequest.getParameter(CODE_PARAMETER);
        final String state = httpRequest.getParameter(STATE_PARAMETER);
        
        //if duo's webSDK becomes OAuth2.0 complaint again, remove this
        final String duoCode = httpRequest.getParameter(DUO_CODE_PARAMETER);
        
        if (state == null || (code == null && duoCode == null)) {
            throw new ExternalAuthenticationException("Duo response must contain a 'code' and 'state' parameter");
        }
        if (code != null && duoCode != null) {
            throw new ExternalAuthenticationException("Duo response can not contain both a 'code' and "
                    + "'duo_code' parameter");
        }
       
        final String key;
        final String nonce;
        try {
            key = DuoSupport.extractKeyFromState(state);
            nonce = DuoSupport.extractNonceFromState(state);
        } catch (final DuoException e) {
            throw new ExternalAuthenticationException("Flow execution key component could not be found in the "
                    + "returned state, unable to resume the flow execution",e);
        }       

        final ProfileRequestContext prc = ExternalAuthentication.getProfileRequestContext(key, httpRequest);
        
        final DuoOIDCAuthenticationContext duoContext = duoContextLookupStrategy.apply(prc);
        if (duoContext == null) {
            log.error("No Duo authentication context to store the Duo 2FA response");
            httpRequest.setAttribute(ExternalAuthentication.AUTHENTICATION_ERROR_KEY, EventIds.INVALID_PROFILE_CTX);
            ExternalAuthentication.finishExternalAuthentication(key, httpRequest, httpResponse);
            return;
            
        }
        duoContext.setAuthorizationCode(code != null ? code : duoCode);
        duoContext.setResponseState(nonce);
        ExternalAuthentication.finishExternalAuthentication(key, httpRequest, httpResponse);
        
    }
    
   

}
