/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.impl;

import java.util.function.BiFunction;

import javax.annotation.Nonnull;

import org.opensaml.profile.context.ProfileRequestContext;

import com.nimbusds.jwt.JWTClaimsSet;

import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCIntegration;
import net.shibboleth.idp.plugin.authn.duo.context.DuoOIDCAuthenticationContext;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.logic.Constraint;

/** 
 * Find the issuer from the {@link DuoOIDCIntegration}. Returns null if not found.
 */
public class DuoIssuerClaimLookupStrategy implements BiFunction<ProfileRequestContext, JWTClaimsSet, String> {
    
    /** HTTPS scheme protocol.*/
    @Nonnull @NotEmpty public static final String HTTPS = "https://";
    
    /** The default issuer path, specific to the v1 Duo flow.*/
    @Nonnull @NotEmpty public static final String DEFAULT_ISSUER_PATH = "/oauth/v1/token"; 
    
    /** The URL path component of the issuer.*/
    @Nonnull @NotEmpty private String issuerPath;
    
    /**
     * Sets the issuer URL path component.
     * 
     * @param path the issuer path
     */
    public void setIssuerPath(@Nonnull @NotEmpty final String path) {            
        issuerPath = Constraint.isNotEmpty(path, "Issuer URL path cannot be null or empty");       
    }

    /** {@inheritDoc} */
    @Override public String apply(@Nonnull final ProfileRequestContext context,
            @Nonnull final JWTClaimsSet cliams) {
        
        final AuthenticationContext authnContext = context.getSubcontext(AuthenticationContext.class);
        if (authnContext == null) {
            return null;
        }

        final DuoOIDCAuthenticationContext duoContext = authnContext.getSubcontext(DuoOIDCAuthenticationContext.class);
        if (duoContext == null) {
            return null;
        }
        final DuoOIDCIntegration duoIntegration = duoContext.getIntegration();
        if (duoIntegration == null) {
            return null;
        }
        return HTTPS+duoIntegration.getAPIHost()+issuerPath;
    }

}
