/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.impl;


import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;

import javax.annotation.Nonnull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.plugin.authn.duo.DefaultDuoOIDCIntegration;
import net.shibboleth.idp.plugin.authn.duo.DuoClientException;
import net.shibboleth.idp.plugin.authn.duo.DuoClientInitializationException;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCClient;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCClientFactory;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCClientRegistry;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCIntegration;
import net.shibboleth.idp.plugin.authn.duo.DuoRegistryException;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.ThreadSafeAfterInit;
import net.shibboleth.utilities.java.support.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * <p>The default Duo Client registry for mapping a {@link DuoOIDCIntegration} to either a new
 * or existing {@link DuoOIDCClient} instance.</p>
 * 
 * <p>Clients are created and registered against a {@link DuoOIDCIntegration}. Once created the client is
 * reused for the lifetime of the IdP. </p>
 * 
 * <p>The {@link DuoOIDCIntegration} should decide it's own 'business key' using the
 * {@link #equals(Object)} and {@link #hashCode()} method appropriately. The {@link DefaultDuoOIDCIntegration} 
 * uses the clientID as its key.</p>
 * 
 * <p>Supports lazy initialization of clients when they are first requested. A single, configurable, client factory 
 * is called to initialize new clients.</p>
 * 
 * <p>Initialization and fetching is thread safe thanks to the use of a {@link ConcurrentMap} and its 
 * {@link ConcurrentMap#computeIfAbsent(Object, Function)} operation. This guarantees that two clients should never
 * be created for the same integration.</p> 
 * 
 */
@ThreadSafeAfterInit
public class DefaultDuoOIDCClientRegistry extends AbstractIdentifiableInitializableComponent
             implements DuoOIDCClientRegistry {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(DefaultDuoOIDCClientRegistry.class);
    
    /** Factory to produce Duo clients.*/
    @NonnullAfterInit private DuoOIDCClientFactory clientFactory;
    
    /** Registry of Duo client to Duo integration.*/
    @Nonnull @NonnullElements private final ConcurrentMap<DuoOIDCIntegration, DuoOIDCClient> clientRegistry;
    
    /** Function for creating a DuoClient from a DuoIntegration. */
    @Nonnull private Function<DuoOIDCIntegration, DuoOIDCClient> clientRegistryMappingFunction; 
    
    /** Constructor.*/
    public DefaultDuoOIDCClientRegistry() {
        clientRegistry = new ConcurrentHashMap<DuoOIDCIntegration, DuoOIDCClient>(1);
        clientRegistryMappingFunction = new CreateNewClientMappingFunction();
    }
    
    /**
     * Set the client factory to use.
     * 
     * @param factory the factory.
     */
    public void setClientFactory(@Nonnull final DuoOIDCClientFactory factory) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        clientFactory = Constraint.isNotNull(factory, "Duo client factory can not be null");
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull public DuoOIDCClient getClientOrCreate(@Nonnull final DuoOIDCIntegration integration) 
            throws DuoRegistryException {
        Constraint.isNotNull(integration, "Duo integration can not be null");

        try {
            //this is an atomic call, avoiding the need to synchronise here e.g. two clients should never 
            //be created for the same integration.
            final DuoOIDCClient client =  clientRegistry.computeIfAbsent(integration,clientRegistryMappingFunction);
            log.debug("Duo registry returning the DuoClient instance '{}' of type '{}'",
                    client.getClientId(),client.getClass().getCanonicalName());
            return client;
        } catch (final DuoClientInitializationException e) {
            throw new DuoRegistryException("DuoClient could not be found or created in the registry",e);
        }        
        
    }
    
    /** {@inheritDoc} */
    @Override protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        if (clientFactory == null) {
            throw new ComponentInitializationException("A Duo Client Factory must be configured and cannot be null");
        }
    }
    
    /**
     * A function for creating a new Duo client from the configured client factory for the given Duo integration.
     * throws a {@link DuoClientInitializationException} if the factory can not create the client.
     */
    private class CreateNewClientMappingFunction implements Function<DuoOIDCIntegration, DuoOIDCClient> {
        
        /** Class logger. */
        @Nonnull private final Logger log = LoggerFactory.getLogger(CreateNewClientMappingFunction.class);
        
        @Override
        @Nonnull public DuoOIDCClient apply(@Nonnull final DuoOIDCIntegration integration){
            
            try {
                log.debug("Creating a new Duo client for integration '{}', using factory type '{}'",integration
                        ,clientFactory.getClass().getTypeName());
                return clientFactory.createInstance(integration);
            } catch (final DuoClientException e) {
                //wrap the exception in a runtime exception.
                throw new DuoClientInitializationException("Could not initialise "
                        + "the DuoClient for the integration with clientId "+integration.getClientId(),e);
            }           
        }
        
    }

}
