/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.audit;

import java.text.ParseException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

import org.opensaml.profile.context.ProfileRequestContext;

import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTClaimsSet;

import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.plugin.authn.duo.context.DuoOIDCAuthenticationContext;


/** 
 * An extension of the {@link AbstractDuoAuditExtractor} that allows extraction of claims inside the 
 * Duo authentication token.
 * 
 * @param <T> the claim type to extract
 */
@ThreadSafe
public abstract class AbstractDuoTokenClaimsAuditExtractor<T> extends AbstractDuoAuditExtractor<T> {
    
    
    @Override
    @Nullable public T apply(@Nullable final ProfileRequestContext profileRequestContext) {
        
        if (profileRequestContext == null) {
            return null;
        }
        final AuthenticationContext authnContext = 
                profileRequestContext.getSubcontext(AuthenticationContext.class);
        if (authnContext == null) {
            return null;
        }
        final DuoOIDCAuthenticationContext duoContext = authnContext.getSubcontext(DuoOIDCAuthenticationContext.class);
        final JWT authToken = duoContext != null ? duoContext.getAuthToken() : null;
        if (authToken == null) {
            return null;
        }
        JWTClaimsSet claimsSet = null;
        try {
            claimsSet = authToken.getJWTClaimsSet();
            if (claimsSet == null) {
                return null;
            }
        } catch (final ParseException e) {
            return null;
        }
        return doLookup(claimsSet, duoContext);
    }
    
    @Override
    @Nullable protected T doLookup(@Nonnull final DuoOIDCAuthenticationContext duoContext) {
        JWTClaimsSet claimsSet = null;
        try {
            final JWT authToken = duoContext.getAuthToken();
            if (authToken == null) {
                return null;
            }
            claimsSet = authToken.getJWTClaimsSet();
            if (claimsSet == null) {
                return null;
            }
        } catch (final ParseException e) {
            return null;
        }
        return doLookup(claimsSet, duoContext);
    }
    
    /**
     * Implemented to perform the actual lookup.
     * 
     * @param claimsSet the claims set of the Duo authentication token
     * @param context the Duo OIDC context
     * 
     * @return lookup value or {@code null} if not found
     */
    @Nullable protected abstract T doLookup(
            final @Nonnull JWTClaimsSet claimsSet, @Nonnull final DuoOIDCAuthenticationContext context);

}
