/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.audit;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.plugin.authn.duo.context.DuoOIDCAuthenticationContext;


/** 
 * An abstract audit extractor function to extract values directly from the Duo context.
 * 
 * @param <T> the claim type to extract
 */
@ThreadSafe
public abstract class AbstractDuoAuditExtractor<T> implements Function<ProfileRequestContext, T> {
    
    
    @Override
    @Nullable public T apply(@Nullable final ProfileRequestContext profileRequestContext) {
        
        if (profileRequestContext == null) {
            return null;
        }
        final AuthenticationContext authnContext = 
                profileRequestContext.getSubcontext(AuthenticationContext.class);
        if (authnContext == null) {
            return null;
        }
        final DuoOIDCAuthenticationContext duoContext = authnContext.getSubcontext(DuoOIDCAuthenticationContext.class);
        if (duoContext == null) {
            return null;
        }        
        return doLookup(duoContext);
    }
    
    /**
     * Implemented to perform the actual lookup.
     * 
     * @param context the Duo OIDC context
     * 
     * @return lookup value or {@code null} if not found
     */
    @Nullable protected abstract T doLookup(@Nonnull final DuoOIDCAuthenticationContext context);

}
