/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo;

import java.util.function.Function;

import javax.annotation.Nonnull;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;

import net.shibboleth.idp.authn.AbstractAuthenticationAction;
import net.shibboleth.idp.authn.AuthnEventIds;
import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.plugin.authn.duo.context.DuoOIDCAuthenticationContext;
import net.shibboleth.shared.annotation.constraint.NonnullBeforeExec;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;

/**
 * <p>A base class for Duo 2FA authentication related actions.</p>
 * 
 * <p>In addition to the work performed by {@link AbstractAuthenticationAction}, this action also looks up
 * and makes available the {@link DuoOIDCAuthenticationContext}.</p>
 * 
 * <p>Duo 2FA authentication action implementations should override the
 * {@link #doExecute(ProfileRequestContext, AuthenticationContext, DuoOIDCAuthenticationContext)} 
 * method.</p>
 * 
 * @event {@link AuthnEventIds#INVALID_AUTHN_CTX}
 * @pre <pre>ProfileRequestContext.getSubcontext(AuthenticationContext.class) != null</pre>
 * @post <pre>AuthenticationContext.getSubcontext(DuoOIDCAuthenticationContext.class) != null</pre>
 */
public abstract class AbstractDuoAuthenticationAction extends AbstractAuthenticationAction {
    
    /** Class logger. */
    @Nonnull @NotEmpty private final Logger log = LoggerFactory.getLogger(AbstractDuoAuthenticationAction.class);
    
    /** Lookup strategy to locate the Duo authentication context. */
    @Nonnull private Function<ProfileRequestContext,DuoOIDCAuthenticationContext> duoContextLookupStrategy;
    
    /** The Duo authentication Context.*/
    @NonnullBeforeExec private DuoOIDCAuthenticationContext duoContext;
        
    
    /** Constructor.*/
    protected AbstractDuoAuthenticationAction() {
        //prc -> ac -> dc
        duoContextLookupStrategy = new ChildContextLookup<>(DuoOIDCAuthenticationContext.class).
                compose(new ChildContextLookup<>(AuthenticationContext.class));
    }
    
    
    /**
     * Set Duo authentication context lookup strategy to use.
     * 
     * @param strategy lookup strategy
     */
    public void setDuoContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,DuoOIDCAuthenticationContext> strategy) {
        ifInitializedThrowUnmodifiabledComponentException();
        ifDestroyedThrowDestroyedComponentException();

        duoContextLookupStrategy = Constraint.isNotNull(strategy, "DuoContextLookuplookup strategy cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected final boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext) {

        if (!super.doPreExecute(profileRequestContext, authenticationContext)) {
            return false;
        }
      
        duoContext = duoContextLookupStrategy.apply(profileRequestContext);
        if (duoContext == null) {
            log.warn("{} No Duo context returned by lookup strategy",getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.INVALID_AUTHN_CTX);
            return false;
            
        }        
        
        return doPreExecute(profileRequestContext, authenticationContext, duoContext);
    }
    
    /**
     * Delegates to {@link #doExecute(ProfileRequestContext, AuthenticationContext, 
     * DuoOIDCAuthenticationContext)} to perform the actual authentication. Implementations can not 
     * override this method.
     * 
     * @param profileRequestContext the current IdP profile request context
     * @param authenticationContext the current authentication context
     */
    @Override
    protected final void doExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext) {
        doExecute(profileRequestContext,authenticationContext,duoContext);
    }
    
    /**
     * Performs this authentication action's pre-execute step. Default implementation just returns true.
     * 
     * @param profileRequestContext the current IdP profile request context
     * @param authenticationContext the current authentication context
     * @param context the Duo authentication context
     * 
     * @return true iff execution should continue
     */
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext,
            @Nonnull final DuoOIDCAuthenticationContext context) {
        return true;
    }
    
    /**
     * Performs this Duo authentication action using the supplied Duo context. Implementations
     * should override this method.
     * 
     * @param profileRequestContext the current IdP profile request context
     * @param authenticationContext the current authentication context
     * @param context the Duo authentication context
     */
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext,
            @Nonnull final DuoOIDCAuthenticationContext context) {
        
    }

}
