/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.model;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.ThreadSafe;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonPOJOBuilder;

import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;

/**
 * <p>Represents a health check response from Duo's 2FA endpoint.</p>
 * 
 * <p>Includes a staged builder for fluent generation. Compatible with Jackson
 * deserialization.</p>
 * 
 * <p>Is immutable, can only be built using the builder and not changed thereafter.</p>
 */
@Immutable
@ThreadSafe
@JsonDeserialize(builder=DuoHealthCheck.Builder.class)
@JsonIgnoreProperties(ignoreUnknown = true)
public final class DuoHealthCheck {
    
    /** A successful 'OK' or unsuccessful 'FAIL' response.*/
    @Nonnull @NotEmpty private final String status;

    /** When a successful response was issued, as seconds since Unix EPOCH.*/
    @Nullable private final DuoHealthCheckResponse response;

    /** The error response code.*/
    @Nullable @NotEmpty private final String code;

    /** When an unsuccessful response was issued, as seconds since Unix EPOCH. */
    @Nullable private final String timestamp;

    /** The error response message.*/
    @Nullable private final String message;

    /** The error response detailed message.*/ 
    @Nullable private final String messageDetail;
    

    /**
     * Private constructor, can only be called by this builder.
     *
     * @param builder the builder to build the instance with
     */
    private DuoHealthCheck(final Builder builder) {
        this.status = builder.status;
        this.response = builder.response;
        this.code = builder.code;
        this.timestamp = builder.timestamp;
        this.message = builder.message;
        this.messageDetail = builder.messageDetail;
    }

    
    /**
     * Get the health status of the 2FA endpoint.
     *
     * @return Returns the status.
     */
    @Nonnull public String getStatus() {
        return status;
    }

    /**
     * Get the response object.
     * 
     * @return Returns the response.
     */
    @Nullable public DuoHealthCheckResponse getResponse() {
        return response;
    }

    /**
     * Get the response code.
     * 
     * @return Returns the code.
     */
    @Nullable public String getCode() {
        return code;
    }

    /**
     * Get the timestamp!!
     * 
     * @return Returns the timestamp.
     */
    @Nullable public String getTimestamp() {
        return timestamp;
    }

    /**
     * Get the response message.
     * 
     * @return Returns the message.
     */
    @Nullable public String getMessage() {
        return message;
    }

    /**
     * Get a detailed response message.
     * 
     * @return Returns the messageDetail.
     */
    @Nullable public String getMessageDetail() {
        return messageDetail;
    }
    
    /** {@inheritDoc} */
    @Override
    public String toString() {
        return String.format(
                "DuoHealthCheck [status=%s, response=%s, code=%s, timestamp=%s, message=%s, messageDetail=%s]",
                status, response, code, timestamp, message, messageDetail);
    }


    /**
     * Creates builder to build {@link DuoHealthCheck}.
     * @return created builder
     */
    
    public static IStatusStage builder() {
        return new Builder();
    }

    /** Status builder.*/
    public interface IStatusStage {
        
        /**
         * Set the status.
         * 
         * @param status the status
         * @return the builder.
         */
        public IBuildStage withStatus(final String status);
    }

    /** Required fields builder.*/
    public interface IBuildStage {
        
        /**
         * Set the response.
         * 
         * @param response the response.
         * 
         * @return the builder.
         */
        public IBuildStage withResponse(final DuoHealthCheckResponse response);

        /**
         * Set the code.
         * 
         * @param code the code
         * @return the builder.
         */
        public IBuildStage withCode(final String code);

        /**
         * Set the timestamp.
         * 
         * @param timestamp the timestamp
         * @return the builder.
         */
        public IBuildStage withTimestamp(final String timestamp);

        /**
         * Set the message.
         * 
         * @param message the message
         * @return the builder.
         */
        public IBuildStage withMessage(final String message);

        /**
         * Set the messageDetail.
         * 
         * @param messageDetail the messageDetail
         * @return the builder.
         */
        public IBuildStage withMessageDetail(final String messageDetail);

        /**
         * Build the health check response.
         * 
         * @return the health check response.
         */
        public DuoHealthCheck build();
    }

    /**
     * Builder to build {@link DuoHealthCheck}.
     */   
    @JsonPOJOBuilder(buildMethodName = "build",withPrefix = "with")
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static final class Builder implements IStatusStage, IBuildStage {
        
        /** The status.*/
        private String status;

        /** The response.*/
        private DuoHealthCheckResponse response;

        /** The code.*/
        private String code;

        /** The timestamp.*/
        private String timestamp;

        /** The message.*/
        private String message;

        /** The message detail.*/
        private String messageDetail;

        private Builder() {
        }

        @Override
        @JsonProperty("stat")
        public IBuildStage withStatus(final String stat) {
            this.status = stat;
            return this;
        }

        @Override
        @JsonProperty("response")
        public IBuildStage withResponse(final DuoHealthCheckResponse resp) {
            this.response = resp;
            return this;
        }

        @Override
        @JsonProperty("code")
        public IBuildStage withCode(final String codeIn) {
            this.code = codeIn;
            return this;
        }

        @Override
        @JsonProperty("timestamp")
        public IBuildStage withTimestamp(final String timestampIn) {
            this.timestamp = timestampIn;
            return this;
        }

        @Override
        @JsonProperty("message")
        public IBuildStage withMessage(final String msg) {
            this.message = msg;
            return this;
        }

        @Override
        @JsonProperty("message_detail")
        public IBuildStage withMessageDetail(final String msgDetail) {
            this.messageDetail = msgDetail;
            return this;
        }

        @Override
        public DuoHealthCheck build() {
            return new DuoHealthCheck(this);
        }
    }

   

}
