/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.context;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;

import org.opensaml.messaging.context.BaseContext;

import com.nimbusds.jwt.JWT;

import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCClient;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCIntegration;

/**
 * <p>Mutable Context that carries the Duo integration, request-response state, authorization code,
 *  and Duo authentication result token.</p>
 *  
 *  <p>As with other context classes, holds state, is not thread-safe and should be thread-confined.</p>
 * 
 * @parent {@link AuthenticationContext}
 * @added After extracting the Duo integration for the given authentication request.
 */
@NotThreadSafe
public final class DuoOIDCAuthenticationContext extends BaseContext {

    /** Username. */
    @Nullable private String username;
    
    /** The selected Duo integration to use for the lifetime of this request.*/
    @Nullable private DuoOIDCIntegration integration;
    
    /** A randomly generated 32 character minimum String sent in the Duo 2FA authorization request.*/
    @Nullable private String requestState;
    
    /** A randomly generated 32 character minimum String returned in the Duo 2FA authorization response.*/
    @Nullable private String responseState;
    
    /** String value used to associate a Client session with an ID Token to mitigate replay attacks.*/ 
    @Nullable private String nonce;
    
    /** The authorization code return from the Duo authorization request.*/
    @Nullable private String authCode;
    
    /** The JWT token received from Duo as a result of 2FA. Token *must* be signed.*/
    @Nullable private JWT authToken;
    
    /** The Duo OIDC client to use for the lifetime of this authentication request.*/
    @Nullable private DuoOIDCClient client;   
    
    /** 
     * A request-bound redirect URI that was determined at runtime. Used to override the URI inside
     * a {@link DuoOIDCIntegration} for supported clients. Allows per-request redirects
     * e.g. useful if one IdP instance is fronted by different virtual hosts.
     */
    @Nullable private String redirectURIOverride;

    /** Public no-arg constructor to allow auto-creation. */
    public DuoOIDCAuthenticationContext() {
        
    }
    
    /**
     * Get the override redirect URI.
     * 
     * @return the override redirect URI.
     */
    @Nullable public String getRedirectURIOverride() {
        return redirectURIOverride;
    }
    
    /**
     * Set the override redirect URI.
     * 
     * @param override the override redirect URI.
     * 
     * @return this context.
     */
    @Nonnull public DuoOIDCAuthenticationContext setRedirectURIOverride(
            @Nullable final String override) {
        redirectURIOverride = override;
        return this;
    }
    
    /** 
     * Get the nonce.
     * 
     * @return the nonce.
     */
    @Nullable public String getNonce() {
        return nonce;
    }
    
    /**
     * Set the OIDC nonce.
     * 
     * @param oidcNonce the nonce.
     * 
     * @return this context.
     */
    @Nonnull public DuoOIDCAuthenticationContext setNonce(@Nullable final String oidcNonce) {
        nonce = oidcNonce;
        return this;
    }
    
    /**
     * Get the authorization code.
     * 
     * @return the authorization code.
     */
    @Nullable public String getAuthorizationCode() {
        return authCode;
    }
    
    /**
     * Set the authorization code.
     * 
     * @param code the authorization code.
     * 
     * @return this context.
     */
    @Nonnull public DuoOIDCAuthenticationContext setAuthorizationCode(@Nullable final String code) {
        authCode = code;
        return this;
    }
    
    /**
     * Get the client used to communicate with the Duo OIDC API.
     * 
     * @return the duo client.
     */
    @Nullable public DuoOIDCClient getClient() {
        return client;
    }
    
    /**
     * Set the client used to communicate with the Duo OIDC API.
     * 
     * @param duoClient the duo client.
     * 
     * @return this context.
     */
    @Nonnull public DuoOIDCAuthenticationContext setClient(@Nullable final DuoOIDCClient duoClient) {
        client = duoClient;
        return this;
    }
    
    /**
     * Set the Duo authentication token.
     * 
     * @param token the duo authentication token
     * 
     * @return this context
     */
    @Nonnull public DuoOIDCAuthenticationContext setAuthToken(@Nullable final JWT token) {
        authToken = token;
        return this;
    }
    
    /**
     * Get the Duo authentication token.
     * 
     * @return the token
     */
    @Nullable public JWT getAuthToken() {
        return authToken;
    }
    
    /**
     * Get the request state.
     * 
     * @return the state
     */
    @Nullable public String getRequestState() {
        return requestState;
    }
    
    
    /**
     * Set the request state.
     * 
     * @param state the state.
     * 
     * @return this context
     */
    @Nonnull public DuoOIDCAuthenticationContext setRequestState(@Nullable final String state) {
        requestState = state;
        return this;
    }
    
    /**
     * Get the response state.
     * 
     * @return the state
     */
    @Nullable public String getResponseState() {
        return responseState;
    }
    
    /**
     * Set the response state.
     * 
     * @param state the state.
     * 
     * @return this context
     */
    @Nonnull public DuoOIDCAuthenticationContext setResponseState(@Nullable final String state) {
        responseState = state;
        return this;
    }
    
    /**
     * Get the username.
     * 
     * @return username
     */
    @Nullable public String getUsername() {
        return username;
    }
    
    /**
     * Set the username.
     * 
     * @param name username
     * 
     * @return this context
     */
    @Nonnull public DuoOIDCAuthenticationContext setUsername(@Nullable final String name) {
        username = name;
        return this;
    }
    
    /**
     * Set the Duo integration.
     * 
     * @param duoIntegration the integration
     * @return this context
     */
    @Nonnull public DuoOIDCAuthenticationContext setIntegration(
                @Nullable final DuoOIDCIntegration duoIntegration) {
        integration = duoIntegration;
        return this;
    }
    
    /**
     * Get the Duo integration.
     * 
     * @return the Duo integration
     */
    @Nullable public DuoOIDCIntegration getIntegration() {
        return integration;
    }

    
}