/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo;

import java.net.URI;
import java.net.URISyntaxException;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;

import org.apache.http.client.utils.URIBuilder;

/** URL support class.*/
@ThreadSafe
public final class URISupport {
    
    /** Private constructor.*/
    private URISupport() {
        
    }
    
    /**
     * Build a {@link URI} from the given parameters. If the scheme is either
     * 'http' or 'https' with their respective default port, the port is set to -1.
     * 
     * @param scheme the scheme
     * @param host the hostname
     * @param port the port
     * @param path the path
     * 
     * @return a fully built URI from the given parameters.
     * 
     * @throws URISyntaxException if the URI can not be constructed.
     */
    @Nonnull public static final URI buildURIIgnoreDefaultPorts(@Nonnull final String scheme, 
            @Nonnull final String host, @Nonnull final int port, 
            @Nonnull final String path) throws URISyntaxException {
        
        int usedPort = port;
        if ("http".equalsIgnoreCase(scheme)) {
            // ignore port iff using the default http port
            if (port == 80) {
                usedPort = -1;
            }
        } else if ("https".equalsIgnoreCase(scheme)) {
            // ignore port iff using the default https port
            if (port == 443) {
                usedPort = -1;
            }
        }
        final URI uri = new URIBuilder()
                .setScheme(scheme)
                .setHost(host)
                .setPort(usedPort)
                .setPath(path)
                .build();
        return uri;
    }
    
    /**
     * Builds the 'origin' (see RFC 6454) from given URI. Omits default or unknown ports. 
     *  
     * @param uri the URI to build the origin from
     * @return the origin
     * @throws URISyntaxException if there is an error getting information from the URI.
     */
    @Nonnull public static String buildOrigin(@Nonnull final URI uri) throws URISyntaxException {    

        if (uri.getPort() == -1) {
            //is the default port (or is not defined), do not include
            return new URI(String.format("%s://%s", uri.getScheme(),uri.getHost())).toString();
        } else {
            return new URI(String.format("%s://%s:%s", uri.getScheme(),uri.getHost(),uri.getPort())).toString();
        }

    }

}
