/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo;

import java.security.Principal;
import java.util.Collection;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import javax.security.auth.Subject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.annotation.constraint.Unmodifiable;
import net.shibboleth.utilities.java.support.component.AbstractInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

/**
 * A data wrapper for use with Duo OIDC integrations which does not support redirectURI generation. 
 */
@ThreadSafe
public final class SimpleDuoOIDCIntegration 
            extends AbstractInitializableComponent implements DuoOIDCIntegration{
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(SimpleDuoOIDCIntegration.class);
    
    /** API host. */
    @GuardedBy("this") @NonnullAfterInit @NotEmpty private String apiHost;
    
    /** Integration key. */
    @GuardedBy("this") @NonnullAfterInit @NotEmpty private String clientId;
    
    /** Secret key. */
    @GuardedBy("this") @NonnullAfterInit @NotEmpty private String secretKey;
    
    /** The used (by clients) redirect_uri to send the client after authorisation .*/
    @GuardedBy("this") @Nullable private String redirectURI;     
    
    /** The URL path to the health endpoint.*/
    @GuardedBy("this") @NonnullAfterInit @NotEmpty private String healthEndpoint;
    
    /** The URL path to the authorization endpoint.*/
    @GuardedBy("this") @NonnullAfterInit @NotEmpty private String authorizeEndpoint;
    
    /** The URL path to the token endpoint.*/
    @GuardedBy("this") @NonnullAfterInit @NotEmpty private String tokenEndpoint;
    
    /** Container for supported principals. */
    @GuardedBy("this") @Nonnull private final Subject supportedPrincipals;
    
    /** Constructor. */
    public SimpleDuoOIDCIntegration() {
        supportedPrincipals = new Subject();
    }
    
    

    @Override
    @NonnullAfterInit @NotEmpty public synchronized String getAPIHost() {
        return apiHost;
    }
    
    /**
     * Set the API host to use.
     * 
     * @param host API host
     */
    public synchronized void setAPIHost(@Nonnull @NotEmpty final String host) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        apiHost = Constraint.isNotNull(StringSupport.trimOrNull(host), "API host cannot be null or empty");
    }
    
    @Override
    @NonnullAfterInit @NotEmpty public synchronized String getHealthCheckEndpoint() {
        return healthEndpoint;
    }
    
    /**
     * Set the health check endpoint URL path.
     * 
     * @param endpoint the endpoint.
     */
    public synchronized void setHealthCheckEndpoint(@Nonnull @NotEmpty final String endpoint) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        healthEndpoint = Constraint.isNotNull(StringSupport.trimOrNull(endpoint), 
                "Health check endpoint cannot be null or empty");
    }

    @Override
    @NonnullAfterInit @NotEmpty public synchronized String getAuthorizeEndpoint() {
        return authorizeEndpoint;
    }
    
    /**
     * Set the authorize endpoint URL path.
     * 
     * @param endpoint the endpoint.
     */
    public synchronized void setAuthorizeEndpoint(@Nonnull @NotEmpty final String endpoint) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        authorizeEndpoint = Constraint.isNotNull(StringSupport.trimOrNull(endpoint), 
                "Authorize endpoint cannot be null or empty");
    }

    @Override
    @NonnullAfterInit @NotEmpty public synchronized String getTokenEndpoint() {
        return tokenEndpoint;
    }
    
    /**
     * Set the token endpoint URL path.
     * 
     * @param endpoint the endpoint.
     */
    public synchronized void setTokenEndpoint(@Nonnull @NotEmpty final String endpoint) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        tokenEndpoint = Constraint.isNotNull(StringSupport.trimOrNull(endpoint), 
                "Token endpoint cannot be null or empty");
    }
    
    @Override
    @Nullable public synchronized String getRedirectURI() {
        return redirectURI;
    }
    
    /**
     * Set the redirect URI.
     * 
     * @param url the url.
     */
    public synchronized void setRedirectURI(@Nonnull final String url) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        redirectURI = Constraint.isNotNull(StringSupport.trimOrNull(url), "Redirect URI cannot be null or empty");;
    }
    
   
    /**
     * Set the client ID to use.
     * 
     * @param id the client identifier.
     */
    public synchronized void setClientId(@Nonnull @NotEmpty final String id) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        clientId = Constraint.isNotNull(StringSupport.trimOrNull(id), "ClientID cannot be null or empty");
    }
    
    @Override
    @NonnullAfterInit @NotEmpty public synchronized String getClientId() {
        return clientId;
    }

    /**
     * Set the secret key to use.
     * 
     * @param key secret key
     */
    public synchronized void setSecretKey(@Nonnull @NotEmpty final String key) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        secretKey = Constraint.isNotNull(StringSupport.trimOrNull(key), "Secret key cannot be null or empty");
    }
    
    @Override
    @NonnullAfterInit @NotEmpty public synchronized String getSecretKey() {
        return secretKey;
    }
    

    @Override
    @Nonnull @NonnullElements @Unmodifiable
    public synchronized <T extends Principal> Set<T> getSupportedPrincipals(@Nonnull final Class<T> c) {
        return supportedPrincipals.getPrincipals(c);
    }
    
    /**
     * Set supported non-user-specific principals that the action will include in the subjects
     * it generates, in place of any default principals from the flow.
     * 
     * <p>Setting to a null or empty collection will maintain the default behavior of relying on the flow.</p>
     * 
     * @param <T> a type of principal to add, if not generic
     * @param principals supported principals to include
     */
    public synchronized <T extends Principal> void setSupportedPrincipals(
            @Nullable @NonnullElements final Collection<T> principals) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        supportedPrincipals.getPrincipals().clear();
        
        if (principals != null && !principals.isEmpty()) {
            supportedPrincipals.getPrincipals().addAll(Set.copyOf(principals));
        }
    }

    @Override
    protected void doInitialize() throws ComponentInitializationException {
        if (getAPIHost() == null || getClientId() == null || getSecretKey() == null 
                ||  getHealthCheckEndpoint() == null || getAuthorizeEndpoint() == null
                || getTokenEndpoint() == null || getRedirectURI() == null) {
            throw new ComponentInitializationException("API host, clientId, secret key,"
                    + "token endpoint, health check endpoint, authorization endpoint, and "
                    + "redirectURI must be set");
        }
    }

    @Override
    public int hashCode() {
        return Objects.hash(getClientId());
    }

    @Override
    public boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final SimpleDuoOIDCIntegration other = (SimpleDuoOIDCIntegration) obj;
        return Objects.equals(getClientId(), other.getClientId());
    }

    @Override
    public String toString() {
        final StringBuilder builder = new StringBuilder();
        builder.append("DefaultDuoOIDCIntegration [apiHost=");
        builder.append(apiHost);
        builder.append(", clientId=");
        builder.append(clientId);
        builder.append(", redirectURI=");
        builder.append(redirectURI);
        builder.append("]");
        return builder.toString();
    }

    
    
    

}
