/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package net.shibboleth.idp.plugin.authn.duo;

import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.annotation.constraint.Unmodifiable;

/**
 * Specialised {@link DuoOIDCIntegration} that allows redirectURIs to be pre-registered or computed dynamically
 * against a list of allowed origins.
 */
public interface DynamicDuoOIDCIntegration extends DuoOIDCIntegration{
    
    /**
     * Get a list of URL origins that are allowed to appear in computed redirect URIs. 
     * 
     * @return a set of unmodifiable allowed origins. 
     *          Never {@literal null} but could be empty.
     */
    @Nonnull @NotLive @Unmodifiable Set<String> getAllowedOrigins();
    
    /**
     * Get the static pre-registered redirect_uri.
     * 
     * @return the pre-registered redirect_uri.
     */
    @Nullable String getRegisteredRedirectURI();
    
    /**
     * Has a redirectURI been pre-registered i.e. set in the properties file. 
     * 
     * @return true iff {@link #getRegisteredRedirectURI()} is not null, false otherwise.
     */
    boolean isRedirectURIPreregistered();
    
    
    /**
     * <p>Set the redirectURI from the one given in a thread-safe way.</p>
     * 
     * <p>Once set, {@link #getRedirectURI()} should return the value of {@literal computedRedirectURI}.</p>
     * 
     * <p>This is a sate changing operation, and should be thread-safe in its operation, as the
     * integration will be shared amongst threads e.g. synchronize the method call.</p>
     * 
     * @param computedRedirectURI the runtime computed redirectURI.
     * 
     */
    void setRedirectURIIfAbsent(@Nonnull @NotEmpty final String computedRedirectURI);
}
