/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.nimbusds.jwt.JWT;

import net.shibboleth.idp.plugin.authn.duo.model.DuoHealthCheck;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;

/**
 * A client for handling Duo OIDC 2FA interactions. Clients are shared and required to be thread-safe.
 */
public interface DuoOIDCClient extends DuoOIDCClientCapabilities{
    
    /**
     * Returns the capabilities of the underlying client.
     *
     * @return interface to access the service's capabilities
     */
    @Nonnull DuoOIDCClientCapabilities getCapabilities();
    
    /**
     * Check the health of the Duo 2FA endpoint and the clients configuration.
     * 
     * @return the heath check response, never {@code null}.
     * 
     * @throws DuoClientException if there is an error returning the health check response.
     */
    @Nonnull DuoHealthCheck healthCheck() throws DuoClientException;    
    
    /**
     * Constructs an authorization redirection URL string with the query parameters required to initiate 
     * a Duo 2FA request.
     * 
     * @param username The user to be authenticated by Duo, never {@code null}.
     * @param state A randomly generated minimum 22 character String, which is relayed back to the client, 
     *          never {@code null}.
     * @param nonce a randomly generated (minimum 22 character) cryptographically secure nonce that is 
     *              replayed in the id_token. Can be {@code null} if not supported by the client as indicated
     *              by the clients described capabilities.
     * @param redirectURIOverride the redirectURI to use in the authorization request. If not {@literal null} 
     *              and the client supports dynamic redirect URIs, it should override any redirect_uri held 
     *              internally to the client e.g. from the {@link DuoOIDCIntegration}. If the client
     *              does not support dynamic redirect URIs, it can be ignored in favour of one held internally.
     * 
     * @return the authorization redirect URL as a string, never {@code null}.
     * 
     * @throws DuoClientException if there is an error creating the authentication URL.
     */
    @Nonnull @NotEmpty String createAuthUrl(@Nonnull @NotEmpty final String username, 
            @Nonnull @NotEmpty final String state, @Nullable final String nonce,
            @Nullable final String redirectURIOverride) throws DuoClientException;
   
    
    /**
     * Exchanges the authorizaton code for a signed Json Web Token (JWT) which contains information pertaining to
     * the authentication. There is no requirement on the client to verify the tokens authenticity or 
     * claims - these should be performed elsewhere in the flow. The JWT **must** be signed. 
     *
     * @param code An authentication identifier which is exchanged (per OAuth2.0 spec) with Duo for a token.
     *              the token can be used to determine if authentication was successful as well as obtain meta-data 
     *              about the authentication, never {@code null}.     *              
     * @param username The user to be authenticated by Duo, never {@code null}.
     * @param redirectURIOverride the redirectURI to use in the code exchange request. If not {@literal null} 
     *              and the client supports dynamic redirect URIs, it should override any redirect_uri held 
     *              internally to the client e.g. from the {@link DuoOIDCIntegration}. If the client
     *              does not support dynamic redirect URIs, it can be ignored in favour of one held internally.
     *
     * @return the **signed** JWT, never {@code null}.
     * 
     * @throws DuoClientException if there is an error exchanging the auth_code for a token result.
     */
    @Nonnull JWT exchangeAuthorizationCodeFor2FAResult(@Nonnull final String code, 
            @Nonnull final String username, @Nullable final String redirectURIOverride) throws DuoClientException;
    
    /**
     * Get an identifier for this client e.g. a UUID. Only for logging and debugging.
     * 
     * @return the client identifier.
     */
    @Nonnull String getClientId();

}
