/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.model;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonPOJOBuilder;

import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;

/**
 * <p>Represents a health check response from Duo's 2FA endpoint.</p>
 * 
 * <p>Includes a staged builder for fluent generation. Compatible with Jackson
 * deserialization.</p>
 * 
 * <p>Is immutable, can only be built using the builder and not changed thereafter.</p>
 */
@Immutable
@JsonDeserialize(builder=DuoHealthCheck.Builder.class)
@JsonIgnoreProperties(ignoreUnknown = true)
public final class DuoHealthCheck {
    
    /** A successful 'OK' or unsuccessful 'FAIL' response.*/
    @Nonnull @NotEmpty private final String status;

    /** When a successful response was issued, as seconds since Unix EPOCH.*/
    @Nullable private final DuoHealthCheckResponse response;

    /** The error response code.*/
    @Nullable @NotEmpty private final String code;

    /** When an unsuccessful response was issued, as seconds since Unix EPOCH. */
    @Nullable private final String timestamp;

    /** The error response message.*/
    @Nullable private final String message;

    /** The error response detailed message.*/ 
    @Nullable private final String messageDetail;

    
    /**
     * Get the health status of the 2FA endpoint.
     *
     * @return Returns the status.
     */
    @Nonnull public String getStatus() {
        return status;
    }

    /**
     * Get the response object.
     * 
     * @return Returns the response.
     */
    @Nullable public DuoHealthCheckResponse getResponse() {
        return response;
    }

    /**
     * Get the response code.
     * 
     * @return Returns the code.
     */
    @Nullable public String getCode() {
        return code;
    }

    /**
     * Get the timestamp!!
     * 
     * @return Returns the timestamp.
     */
    @Nullable public String getTimestamp() {
        return timestamp;
    }

    /**
     * Get the response message.
     * 
     * @return Returns the message.
     */
    @Nullable public String getMessage() {
        return message;
    }

    /**
     * Get a detailed response message.
     * 
     * @return Returns the messageDetail.
     */
    @Nullable public String getMessageDetail() {
        return messageDetail;
    }
    
    /** {@inheritDoc} */
    @Override
    public String toString() {
        return String.format(
                "DuoHealthCheck [status=%s, response=%s, code=%s, timestamp=%s, message=%s, messageDetail=%s]",
                status, response, code, timestamp, message, messageDetail);
    }

    

    /**
     * Private constructor, can only be called by this builder.
     *
     * @param builder the builder to build the instance with
     */
    private DuoHealthCheck(Builder builder) {
        this.status = builder.status;
        this.response = builder.response;
        this.code = builder.code;
        this.timestamp = builder.timestamp;
        this.message = builder.message;
        this.messageDetail = builder.messageDetail;
    }

    /**
     * Creates builder to build {@link DuoHealthCheck}.
     * @return created builder
     */
    
    public static IStatusStage builder() {
        return new Builder();
    }

    
    public interface IStatusStage {
        public IBuildStage withStatus(String status);
    }

    
    public interface IBuildStage {
        public IBuildStage withResponse(DuoHealthCheckResponse response);

        public IBuildStage withCode(String code);

        public IBuildStage withTimestamp(String timestamp);

        public IBuildStage withMessage(String message);

        public IBuildStage withMessageDetail(String messageDetail);

        public DuoHealthCheck build();
    }

    /**
     * Builder to build {@link DuoHealthCheck}.
     */   
    @JsonPOJOBuilder(buildMethodName = "build",withPrefix = "with")
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static final class Builder implements IStatusStage, IBuildStage {
        private String status;

        private DuoHealthCheckResponse response;

        private String code;

        private String timestamp;

        private String message;

        private String messageDetail;

        private Builder() {
        }

        @Override
        @JsonProperty("stat")
        public IBuildStage withStatus(String status) {
            this.status = status;
            return this;
        }

        @Override
        @JsonProperty("response")
        public IBuildStage withResponse(DuoHealthCheckResponse response) {
            this.response = response;
            return this;
        }

        @Override
        @JsonProperty("code")
        public IBuildStage withCode(String code) {
            this.code = code;
            return this;
        }

        @Override
        @JsonProperty("timestamp")
        public IBuildStage withTimestamp(String timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        @Override
        @JsonProperty("message")
        public IBuildStage withMessage(String message) {
            this.message = message;
            return this;
        }

        @Override
        @JsonProperty("message_detail")
        public IBuildStage withMessageDetail(String messageDetail) {
            this.messageDetail = messageDetail;
            return this;
        }

        @Override
        public DuoHealthCheck build() {
            return new DuoHealthCheck(this);
        }
    }

   

}
