/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo;


import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.idp.authn.principal.PrincipalSupportingComponent;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.annotation.constraint.Unmodifiable;

/**
 * Interface to a particular Duo OIDC integration point. In part replaces
 * OIDC metadata as that is not supported by Duo.
 */
public interface DuoOIDCIntegration extends PrincipalSupportingComponent {
    
    /**
     * Get the name of the API host to contact.
     * 
     * @return name of API host
     */
    @Nonnull @NotEmpty String getAPIHost();

    /**
     * Get the clientId key.
     * 
     * @return the integration key
     */
    @Nonnull @NotEmpty String getClientId();

    /**
     * Get the secret key.
     * 
     * @return the secret key
     */
    @Nonnull @NotEmpty String getSecretKey();
    
    /**
     * Get a list of origins that allowed to appear in computed redirect URIs. 
     * 
     * @return a set of allowed origins. Never {@literal null} but could be empty.
     */
    @Nonnull @NotLive @Unmodifiable Set<String> getAllowedOrigins();
    
    /**
     * <p>Set the redirectURI from one in a thread-safe way.</p>
     * 
     * <p>This is a sate changing operation, and should be thread-safe in its operation, as the
     * integration will be shared amongst threads e.g. syncrhonize the method call.</p>
     * 
     * @param computedRedirectURI the runtime computed redirectURI.
     * 
     */
    void setRedirectURIIfAbsent(@Nonnull @NotEmpty final String computedRedirectURI);
    
    /**
     * Get the runtime redirectURI to direct the client to after authorisation. 
     * 
     * @return the redirectURI
     */
    @Nullable String getRedirectURI();
    
    /**
     * Get the static pre-registered redirect_uri.
     * 
     * @return the pre-registered redirect_uri.
     */
    @Nullable String getRegisteredRedirectURI();
    
    /**
     * Has a redirectURI been pre-registered i.e. set in the properties file. 
     * 
     * @return true iff {@link #getRegisteredRedirectURI()} is not null, false otherwise.
     */
    @Nullable boolean isRedirectURIPreregistered();
    
    /**
     * Get the path of the health check endpoint.
     * 
     * @return the path of the health check endpoint
     */
    @Nonnull @NotEmpty String getHealthCheckEndpoint();
    
    /**
     * Get the path of the authorization endpoint.
     * 
     * @return the path of the authorization endpoint
     */
    @Nonnull @NotEmpty String getAuthorizeEndpoint();
    
    /**
     * Get the path of the token endpoint.
     * 
     * @return the path of the token endpoint;
     */
    @Nonnull @NotEmpty String getTokenEndpoint();

    


}
