/*
 * Decompiled with CFR 0.152.
 */
package io.jenkins.plugins.wiz;

import hudson.AbortException;
import hudson.FilePath;
import hudson.ProxyConfiguration;
import hudson.model.TaskListener;
import hudson.remoting.Callable;
import io.jenkins.plugins.wiz.PGPVerifier;
import io.jenkins.plugins.wiz.ParsedWizCliUrl;
import io.jenkins.plugins.wiz.WizCliSetup;
import io.jenkins.plugins.wiz.WizInputValidator;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.util.logging.Level;
import java.util.logging.Logger;
import jenkins.model.Jenkins;
import jenkins.security.MasterToSlaveCallable;
import org.apache.commons.lang3.SystemUtils;

public class WizCliDownloader {
    private static final Logger LOGGER = Logger.getLogger(WizCliDownloader.class.getName());
    private static final int DOWNLOAD_TIMEOUT = 60000;
    private static final int CONNECT_TIMEOUT = 10000;
    private static final String PUBLIC_KEY_RESOURCE = "/io/jenkins/plugins/wiz/public_key.asc";

    public static WizCliSetup setupWizCli(FilePath workspace, String wizCliURL, TaskListener listener) throws IOException {
        try {
            ParsedWizCliUrl parsedUrl = WizInputValidator.parseWizCliUrl(wizCliURL);
            String[] osDetails = (String[])workspace.act((Callable)new MasterToSlaveCallable<String[], IOException>(){

                public String[] call() {
                    boolean isWindows = SystemUtils.IS_OS_WINDOWS;
                    return new String[]{String.valueOf(isWindows)};
                }
            });
            boolean isWindows = Boolean.parseBoolean(osDetails[0]);
            String cliFileName = isWindows ? "wizcli.exe" : "wizcli";
            FilePath cliPath = workspace.child(cliFileName);
            WizCliDownloader.downloadAndVerifyWizCli(parsedUrl.getUrl(), cliPath, workspace, listener);
            if (!isWindows) {
                cliPath.chmod(493);
            }
            return new WizCliSetup(isWindows, parsedUrl.getVersion());
        }
        catch (AbortException e) {
            listener.error("Invalid Wiz CLI URL format: " + e.getMessage());
            throw e;
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void downloadAndVerifyWizCli(String wizCliURL, FilePath cliPath, FilePath workspace, TaskListener listener) throws IOException {
        try {
            listener.getLogger().println("Downloading Wiz CLI from: " + wizCliURL);
            WizCliDownloader.downloadFile(wizCliURL, cliPath);
            listener.getLogger().println("Download completed successfully");
            String sha256URL = wizCliURL + "-sha256";
            String signatureURL = sha256URL + ".sig";
            FilePath sha256File = workspace.child("wizcli-sha256");
            FilePath signatureFile = workspace.child("wizcli-sha256.sig");
            FilePath publicKeyFile = workspace.child("public_key.asc");
            try {
                WizCliDownloader.downloadFile(sha256URL, sha256File);
                WizCliDownloader.downloadFile(signatureURL, signatureFile);
                WizCliDownloader.extractPublicKey(publicKeyFile);
                WizCliDownloader.verifySignatureAndChecksum(listener, cliPath, sha256File, signatureFile, publicKeyFile, workspace);
            }
            finally {
                WizCliDownloader.cleanupVerificationFiles(workspace, listener);
            }
        }
        catch (Exception e) {
            listener.error("Failed to download or verify Wiz CLI: " + e.getMessage());
            throw new AbortException("Failed to setup Wiz CLI: " + e.getMessage());
        }
    }

    private static void extractPublicKey(FilePath publicKeyFile) throws IOException {
        try (InputStream keyStream = WizCliDownloader.class.getResourceAsStream(PUBLIC_KEY_RESOURCE);){
            if (keyStream == null) {
                throw new IOException("Could not find public key resource");
            }
            String publicKey = new String(keyStream.readAllBytes(), StandardCharsets.UTF_8);
            publicKeyFile.write(publicKey, StandardCharsets.UTF_8.name());
            LOGGER.log(Level.FINE, "Public key extracted successfully");
        }
        catch (Exception e) {
            LOGGER.log(Level.SEVERE, "Failed to extract public key", e);
            throw new IOException("Failed to extract public key from resources", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void downloadFile(String fileURL, FilePath targetPath) throws IOException {
        URL url = new URL(fileURL);
        HttpURLConnection conn = null;
        InputStream inputStream = null;
        OutputStream outputStream = null;
        try {
            ProxyConfiguration proxyConfig = Jenkins.get().getProxy();
            try {
                conn = (HttpURLConnection)(proxyConfig != null ? url.openConnection(proxyConfig.createProxy(url.getHost())) : url.openConnection());
            }
            catch (IllegalArgumentException e) {
                throw new IOException("Invalid proxy configuration", e);
            }
            conn.setConnectTimeout(10000);
            conn.setReadTimeout(60000);
            int responseCode = conn.getResponseCode();
            if (responseCode != 200) {
                throw new IOException("Download failed with HTTP code: " + responseCode);
            }
            FilePath parent = targetPath.getParent();
            if (parent == null) {
                throw new IOException("Invalid target path: parent directory is null");
            }
            try {
                parent.mkdirs();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new IOException("Directory creation was interrupted", e);
            }
            inputStream = conn.getInputStream();
            try {
                int bytesRead;
                outputStream = targetPath.write();
                byte[] buffer = new byte[8192];
                while ((bytesRead = inputStream.read(buffer)) != -1) {
                    outputStream.write(buffer, 0, bytesRead);
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new IOException("File download was interrupted", e);
            }
        }
        finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                }
                catch (IOException e) {
                    LOGGER.log(Level.WARNING, "Error closing input stream", e);
                }
            }
            if (outputStream != null) {
                try {
                    outputStream.close();
                }
                catch (IOException e) {
                    LOGGER.log(Level.WARNING, "Error closing output stream", e);
                }
            }
            if (conn != null) {
                conn.disconnect();
            }
        }
    }

    private static void verifySignatureAndChecksum(TaskListener listener, FilePath cliPath, FilePath sha256File, FilePath signaturePath, FilePath publicKeyPath, FilePath workspace) throws IOException {
        try {
            boolean verified = (Boolean)workspace.act((Callable)new VerifySignatureCallable(sha256File, signaturePath, publicKeyPath));
            if (!verified) {
                throw new IOException("GPG signature verification failed");
            }
            WizCliDownloader.verifyChecksum(cliPath, sha256File);
            listener.getLogger().println("Successfully verified Wiz CLI signature and checksum");
        }
        catch (Exception e) {
            throw new IOException("GPG signature verification failed: " + e.getMessage(), e);
        }
    }

    private static void verifyChecksum(FilePath cliPath, FilePath sha256File) throws IOException, InterruptedException {
        String actualHash;
        String expectedHash = sha256File.readToString().trim();
        if (!expectedHash.equals(actualHash = WizCliDownloader.calculateSHA256(cliPath))) {
            throw new IOException("SHA256 checksum verification failed. Expected: " + expectedHash + ", Actual: " + actualHash);
        }
    }

    private static String calculateSHA256(FilePath filePath) throws IOException {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] buffer = new byte[8192];
            try (InputStream inputStream = filePath.read();){
                int bytesRead;
                while ((bytesRead = inputStream.read(buffer)) != -1) {
                    digest.update(buffer, 0, bytesRead);
                }
            }
            byte[] hash = digest.digest();
            StringBuilder hexString = new StringBuilder();
            for (byte b : hash) {
                String hex = Integer.toHexString(0xFF & b);
                if (hex.length() == 1) {
                    hexString.append('0');
                }
                hexString.append(hex);
            }
            return hexString.toString();
        }
        catch (Exception e) {
            throw new IOException("Failed to calculate SHA256: " + e.getMessage(), e);
        }
    }

    private static void cleanupVerificationFiles(FilePath workspace, TaskListener listener) {
        FilePath[] filesToClean;
        for (FilePath file : filesToClean = new FilePath[]{workspace.child("wizcli-sha256"), workspace.child("wizcli-sha256.sig"), workspace.child("public_key.asc")}) {
            try {
                if (!file.exists()) continue;
                file.delete();
                LOGGER.log(Level.FINE, "Deleted verification file: {0}", file.getRemote());
            }
            catch (Exception e) {
                LOGGER.log(Level.WARNING, "Failed to delete verification file: " + file.getRemote(), e);
                listener.getLogger().println("Warning: Failed to delete " + file.getRemote());
            }
        }
    }

    private static class VerifySignatureCallable
    extends MasterToSlaveCallable<Boolean, IOException> {
        private final FilePath sha256File;
        private final FilePath signaturePath;
        private final FilePath publicKeyPath;

        public VerifySignatureCallable(FilePath sha256File, FilePath signaturePath, FilePath publicKeyPath) {
            this.sha256File = sha256File;
            this.signaturePath = signaturePath;
            this.publicKeyPath = publicKeyPath;
        }

        public Boolean call() throws IOException {
            try {
                PGPVerifier verifier = new PGPVerifier();
                return verifier.verifySignatureFromFiles(this.sha256File.getRemote(), this.signaturePath.getRemote(), this.publicKeyPath.getRemote());
            }
            catch (PGPVerifier.PGPVerificationException e) {
                throw new IOException("PGP verification failed", e);
            }
        }
    }
}

