/*
 * Decompiled with CFR 0.152.
 */
package io.jenkins.plugins.swarmcloud;

import com.github.dockerjava.api.model.Service;
import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import hudson.Extension;
import hudson.model.Descriptor;
import hudson.model.TaskListener;
import hudson.slaves.ComputerLauncher;
import hudson.slaves.JNLPLauncher;
import hudson.slaves.SlaveComputer;
import io.jenkins.plugins.swarmcloud.SwarmAgent;
import io.jenkins.plugins.swarmcloud.SwarmCloud;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import jenkins.model.Jenkins;
import jenkins.slaves.JnlpSlaveAgentProtocol;

public class SwarmComputerLauncher
extends JNLPLauncher {
    private static final Logger LOGGER = Logger.getLogger(SwarmComputerLauncher.class.getName());
    private static final int DEFAULT_TIMEOUT_SECONDS = 300;
    private static final int CHECK_INTERVAL_MS = 5000;
    private final String cloudName;
    private final String image;
    private final boolean useWebSocket;
    private final String workDir;
    private final int connectionTimeoutSeconds;

    @SuppressFBWarnings(value={"NP_METHOD_PARAMETER_TIGHTENS_ANNOTATION"}, justification="Convenience constructor delegates to main constructor with null defaults")
    public SwarmComputerLauncher(@NonNull String cloudName, @NonNull String image) {
        this(cloudName, image, true, null, null, 300);
    }

    public SwarmComputerLauncher(@NonNull String cloudName, @NonNull String image, boolean useWebSocket, @CheckForNull String tunnel, @CheckForNull String workDir) {
        this(cloudName, image, useWebSocket, tunnel, workDir, 300);
    }

    public SwarmComputerLauncher(@NonNull String cloudName, @NonNull String image, boolean useWebSocket, @CheckForNull String tunnel, @CheckForNull String workDir, int connectionTimeoutSeconds) {
        super(tunnel, null);
        this.cloudName = cloudName;
        this.image = image;
        this.useWebSocket = useWebSocket;
        this.workDir = workDir;
        this.connectionTimeoutSeconds = connectionTimeoutSeconds > 0 ? connectionTimeoutSeconds : 300;
    }

    public void launch(SlaveComputer computer, TaskListener listener) {
        PrintStream logger = listener.getLogger();
        try {
            if (!(computer instanceof SwarmAgent.SwarmComputer)) {
                throw new IllegalArgumentException("Expected SwarmComputer, got: " + computer.getClass().getName());
            }
            SwarmAgent.SwarmComputer swarmComputer = (SwarmAgent.SwarmComputer)computer;
            SwarmAgent agent = swarmComputer.getNode();
            if (agent == null) {
                logger.println("ERROR: Agent node is null");
                LOGGER.log(Level.SEVERE, "Agent node is null for computer: {0}", computer.getName());
                return;
            }
            logger.println("=== Swarm Agent Launch ===");
            logger.println("Agent name: " + agent.getNodeName());
            logger.println("Service ID: " + agent.getServiceId());
            logger.println("Image: " + this.image);
            logger.println("Connection mode: " + (this.useWebSocket ? "WebSocket" : "JNLP/TCP"));
            LOGGER.log(Level.FINE, "Launching Swarm agent: {0}, service: {1}, webSocket: {2}", new Object[]{agent.getNodeName(), agent.getServiceId(), this.useWebSocket});
            logger.println("Waiting for agent to connect (timeout: " + this.connectionTimeoutSeconds + "s)...");
            this.waitForConnection(computer, listener, this.connectionTimeoutSeconds);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            logger.println("ERROR: Launch interrupted");
            LOGGER.log(Level.WARNING, "Launch interrupted for: " + computer.getName(), e);
        }
        catch (Exception e) {
            logger.println("ERROR: " + e.getMessage());
            LOGGER.log(Level.SEVERE, "Launch failed for: " + computer.getName(), e);
        }
    }

    @SuppressFBWarnings(value={"RCN_REDUNDANT_NULLCHECK_OF_NONNULL_VALUE"}, justification="Defensive null check for Jenkins API method")
    private void waitForConnection(SlaveComputer computer, TaskListener listener, int timeoutSeconds) throws IOException, InterruptedException {
        SwarmCloud cloud;
        PrintStream logger = listener.getLogger();
        long startTime = System.currentTimeMillis();
        long timeoutMs = (long)timeoutSeconds * 1000L;
        int checkCount = 0;
        while (System.currentTimeMillis() - startTime < timeoutMs) {
            SwarmCloud cloud2;
            SwarmAgent agent;
            String offlineCause;
            long elapsed;
            if (computer.isOnline()) {
                elapsed = (System.currentTimeMillis() - startTime) / 1000L;
                logger.println("Agent connected successfully after " + elapsed + " seconds");
                LOGGER.log(Level.FINE, "Agent connected: {0} in {1}s", new Object[]{computer.getName(), elapsed});
                return;
            }
            if (computer.isOffline() && (offlineCause = computer.getOfflineCauseReason()) != null && !offlineCause.isEmpty() && !offlineCause.contains("Waiting for")) {
                LOGGER.log(Level.WARNING, "Agent offline with cause: {0}", offlineCause);
            }
            if (++checkCount % 6 == 0 && (agent = (SwarmAgent)computer.getNode()) != null && (cloud2 = agent.getCloud()) != null) {
                this.checkServiceHealth(cloud2, agent.getServiceId(), logger);
            }
            Thread.sleep(5000L);
            elapsed = (System.currentTimeMillis() - startTime) / 1000L;
            if (elapsed % 30L != 0L) continue;
            logger.println("Still waiting for agent to connect... (" + elapsed + "s elapsed)");
        }
        String errorMsg = "Timeout waiting for agent to connect after " + timeoutSeconds + " seconds";
        LOGGER.log(Level.SEVERE, errorMsg);
        logger.println("ERROR: " + errorMsg);
        SwarmAgent agent = (SwarmAgent)computer.getNode();
        if (agent != null && (cloud = agent.getCloud()) != null) {
            try {
                String logs = cloud.getDockerClient().getServiceLogs(agent.getServiceId(), 50);
                if (logs != null && !logs.isEmpty()) {
                    logger.println("=== Service Logs ===");
                    logger.println(logs);
                }
            }
            catch (Exception e) {
                logger.println("Could not retrieve service logs: " + e.getMessage());
            }
        }
        throw new IOException(errorMsg);
    }

    private void checkServiceHealth(SwarmCloud cloud, String serviceId, PrintStream logger) {
        try {
            Service service = cloud.getDockerClient().getService(serviceId);
            if (service == null) {
                logger.println("WARNING: Service not found - may have been terminated");
            }
        }
        catch (Exception e) {
            LOGGER.log(Level.FINE, "Error checking service health", e);
        }
    }

    public void afterDisconnect(SlaveComputer computer, TaskListener listener) {
        LOGGER.log(Level.FINE, "Agent disconnected: {0}", computer.getName());
    }

    public void beforeDisconnect(SlaveComputer computer, TaskListener listener) {
        LOGGER.log(Level.FINE, "Agent about to disconnect: {0}", computer.getName());
    }

    @NonNull
    public static String getAgentSecret(@NonNull String agentName) {
        Jenkins jenkins = Jenkins.getInstanceOrNull();
        if (jenkins == null) {
            throw new IllegalStateException("Jenkins instance is not available");
        }
        return JnlpSlaveAgentProtocol.SLAVE_SECRET.mac(agentName);
    }

    @NonNull
    public static String[] buildAgentCommand(@NonNull String jenkinsUrl, @NonNull String agentName, @NonNull String secret, boolean useWebSocket, @CheckForNull String workDir) {
        ArrayList<String> args = new ArrayList<String>();
        args.add("-url");
        args.add(jenkinsUrl);
        if (useWebSocket) {
            args.add("-webSocket");
        }
        args.add("-name");
        args.add(agentName);
        args.add("-secret");
        args.add(secret);
        if (workDir != null && !workDir.isBlank()) {
            args.add("-workDir");
            args.add(workDir);
        }
        return args.toArray(new String[0]);
    }

    @NonNull
    public static Map<String, String> buildAgentEnvironment(@NonNull String jenkinsUrl, @NonNull String agentName, @NonNull String secret, boolean useWebSocket, @CheckForNull String workDir) {
        LinkedHashMap<String, String> env = new LinkedHashMap<String, String>();
        env.put("JENKINS_URL", jenkinsUrl);
        env.put("JENKINS_AGENT_NAME", agentName);
        env.put("JENKINS_SECRET", secret);
        if (useWebSocket) {
            env.put("JENKINS_WEB_SOCKET", "true");
        }
        if (workDir != null && !workDir.isBlank()) {
            env.put("JENKINS_AGENT_WORKDIR", workDir);
        }
        env.put("JENKINS_DIRECT_CONNECTION", jenkinsUrl.replace("http://", "").replace("https://", ""));
        return env;
    }

    @NonNull
    public static String getJnlpUrl(@NonNull String jenkinsUrl, @NonNull String agentName) {
        Object baseUrl = jenkinsUrl.endsWith("/") ? jenkinsUrl : jenkinsUrl + "/";
        return (String)baseUrl + "computer/" + agentName + "/jenkins-agent.jnlp";
    }

    public String getCloudName() {
        return this.cloudName;
    }

    public String getImage() {
        return this.image;
    }

    public boolean isUseWebSocket() {
        return this.useWebSocket;
    }

    public String getWorkDir() {
        return this.workDir;
    }

    public int getConnectionTimeoutSeconds() {
        return this.connectionTimeoutSeconds;
    }

    @Extension
    public static class DescriptorImpl
    extends Descriptor<ComputerLauncher> {
        @NonNull
        public String getDisplayName() {
            return "Docker Swarm Agent Launcher";
        }
    }
}

